///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava;



/******************************************************************************
Indicates unreachable execution path.

<p><b>Details:</b> An <code>UnreachableCodeException</code> is thrown when an execution path that the programmer has assumed to be unreachable actually turns out to be reachable.  Therefore, ideally, this exception should never be thrown, and if it is, it indicates a flaw in the programmer's thinking -- but this is better than allowing the flaw to go undetected!</p>

<p>As an extra safeguard to your program, always be sure to throw this exception wherever you have assumed that code is unreachable.  This will help you debug your program later if it turns out that you were wrong.</p>

<ul>
	<p><b>Example:</b></p>

	<p>Your code <code>switch</code>es on the variable <code>shape</code>.  You know, in this particular <code>switch</code> statement, that <var>shape</var> can be only one of three values.  Your code might look like this:</p>

	<ul><pre>
	*int sides;
	*switch (shape)
	*{
	*case TRIANGLE:
	*	sides = 3;
	*	break;
	*case SQUARE:
	*	sides = 4;
	*	break;
	*case PENTAGON:
	*	sides = 5;
	*	break;
	*default:
	*	<i>// We know that shape must have been one of
	*	// the three values mentioned above, but we're
	*	// putting a catch-all in here, just in case.</i>
	*	<b>throw new lava.UnreachableCodeException ();</b>
	*}
	*return sides;
	</pre></ul>

	<p>In this example, adding the unreachable <code>default</code> case not only guarantees that you'll be notified if your assumption is wrong, but, in this case, <em>allows the compiler to process your code without complaining that <code>sides</code> might not have been initialized</em>.  (You see, compilers just aren't as smart as humans!)</p>
</ul>

<p>There are times when an <code>UnreachableCodeException</code> should be used just to fool the compiler:</p>

<ul>
	<p><b>Example:</b></p>

	<p>The method <code>foo</code> divides a value by two if the value is even, but terminates the program if the value is odd.</p>

	<ul><pre>
	*int foo (int n)
	*{
	*	if (n % 2 == 0)
	*		return n / 2;
	*	System.exit (0);
	*	<i>// exit() will never return, but does the compiler
	*	// know that?  No!  It will complain about needing
	*	// a return value right here.</i>
	*}
	</pre></ul>

	<p>Although this code looks correct, because the <code>exit</code> method does not return, the compiler will not accept it.  This is because the compiler is not aware that the call to <code>System.exit</code> will never return.  Instead, it expects a return value at the end of the method.  This problem is easily remedied, in a self-documenting way, by using <code>UnreachableCodeException</code>.</p>

	<ul><pre>
	*int foo (int n)
	*{
	*	if (n % 2 == 0)
	*		return n / 2;
	*	System . exit (0);
	*	<b>throw new lava.UnreachableCodeException ();</b>
	*}
	</pre></ul>

	<p>The compiler will now accept this code.  Although you could have inserted a dummy <code>return</code> statement into the code, the approach shown here renders the code more "self-documenting," and guarantees that you'll be notified if your assumption about <code>System.exit</code> ever turns out to be <code>false</code> (for some strange reason).</p>

</ul>

@since 1998
@version 1999.04.12
******************************************************************************/

public class UnreachableCodeException
	extends RuntimeException
{



	/**********************************************************************
	Initializes without message.

	<p><b>Details:</b>  This default constructor initializes a new <code>UnreachableCodeException</code> without an exception message.</p>
	**********************************************************************/

	public UnreachableCodeException ()
	{
		super ("unreachable code");
	}



	/**********************************************************************
	Initializes with message.

	<p><b>Details:</b>  This constructor initializes a new <code>UnreachableCodeException</code> with the given exception message.</p>

	@param s the messsage
	**********************************************************************/

	public UnreachableCodeException (String s)
	{
		super (s);
	}



}



