///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava . clib;



/******************************************************************************
Ports of ctype functions.

<p><b>Details:</b>  <code>Ctype</code> is port of the <tt>ctype</tt> library from the C programming language.  By emulating many of the C language's <tt>ctype</tt> library functions, this class lets die-hard C programmers write Java code without completely converting to the Java API.  Rather than wasting valuable time figuring out how to perform the Java equivalent of tasks that are easy in C, the programmer can simply use the methods in this class instead.</p>

<p>This class also contains some additional functions that have similar features.</p>
******************************************************************************/

public final class Ctype
{



	/**********************************************************************
	Returns <code>true</code> if c is a letter.  Returns the result of <code>Character.isLetter(c)</code>.
	**********************************************************************/

	public static boolean isalpha (int c)
	{
		return Character.isLetter ((char) c);
	}



	/**********************************************************************
	Returns <code>true</code> if c is a decimal digit.  Returns the result of <code>Character.isDigit(c)</code>.
	**********************************************************************/

	public static boolean isdigit (int c)
	{
		return Character.isDigit ((char) c);
	}



	/**********************************************************************
	Returns <code>true</code> if c is alphanumeric.  Specifically, returns the result of <code>Character.isLetterOrDigit(c)</code>.
	**********************************************************************/

	public static boolean isalnum (int c)
	{
		return Character.isLetterOrDigit ((char) c);
	}



	/**********************************************************************
	Returns <code>true</code> if c is a character that creates "white space" in displayed text.  Specifically, returns <code>Character.isWhitespace(c)</code>.
	**********************************************************************/

	public static boolean isspace (int c)
	{
		return Character.isWhitespace ((char) c);
	}



	/**********************************************************************
	Tests for vertical space.

	<p><b>Details:</b>  isvspace returns true if the given character (<var>c</var>) is a vertical space.  The criteria for determining this is as follows (in order of precedence):</p>

	<ol>
		<li>Characters '\n', '\v', '\f', '\r' are vertical spaces.
		<li>Other characters below 256 are not.
		<li>No character that is not a white space according to Character.isWhitespace can be a vertical space.
		<li>
			Characters belonging to the unicode types:
			<ul>
				<li>LINE_SEPARATOR
				<li>PARAGRAPH_SEPARATOR
			</ul>
			are vertical spaces.
		<li>All other characters are not vertical spaces.
	</ol>

	@param c the character to test
	@return true iff c is a vertical space

	@since 1998
	@version 2000.06.03
	**********************************************************************/

	public static boolean isvspace (int c)
	{
		// Handle common cases first:
		switch (c)
		{
		case '\n':
		case 11: // '\v'
		case '\f':
		case '\r':
			return true;
		default:
			if (c < 256)
				return false;
		}
		if (! isspace (c))
			return false;
		switch (Character.getType ((char) c))
		{
		case Character.LINE_SEPARATOR:
		case Character.PARAGRAPH_SEPARATOR:
			return true;
		}
		return false;
	}



	/**********************************************************************
	Tests for horizontal space.

	<p><b>Details:</b>  ishspace returns true if and only if the given character is a horizontal space.  The test is conducted as follows:</p>

	<ol>
		<li>Characters '&nbsp;' and '\t' are horizontal spaces.
		<li>All other characters, including the so-<wbr>called no-<wbr>break spaces, are not.
	</ol>

	<p>There may be other unicode characters that can be considered horizontal spaces.  If you discover any, please let the author know and the test will be patched in a future release.</p>

	@param c the character to test
	@return true iff c is a horizontal space

	@since 1998
	@version 2000.06.03
	**********************************************************************/

	public static boolean ishspace (int c)
	{
		switch (c)
		{
		case ' ':
		case '\t':
			return true;
		}
		return false;
	}



}



