///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import lava.Platform;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;



/******************************************************************************
ASCII output stream.

<p><b>Details:</b>  <code>AsciiOutputStream</code> is an <code>OutputStream</code> with extensions for writing <code>String</code>s in 8-bit ASCII regardless of the platform's character encoding.  In the <code>write</code> and <code>writeln</code> methods, which write <code>String</code>s, each character in the string is cast directly into a single <code>byte</code>, using only the low order 8 bits, before it is written.</p>

<p>This class is especially useful in internationalizable code that, for some functions, must speak ASCII in order to work properly.</p>

@since 2000.01.20
@author Sharky
******************************************************************************/

public class AsciiOutputStream
	extends FilterOutputStream
{



	/**********************************************************************
	Sets forwarding OutputStream.

	<p><b>Details:</b>  This constructor binds this <code>AsciiOutputStream</code> to the given <code>OutputStream</code> (<var>out</var>).  All output produced by this <code>AsciiOutputStream</code> will be forwarded to <var>out</var>.</p>

	@param out the target output stream
	**********************************************************************/

	public AsciiOutputStream (OutputStream out)
	{
		super (out);
	}



	/**********************************************************************
	Current line separator.

	<p><b>Details:</b>  <code>line_separator</code> is the current line separator used by the <code>writeln</code> method.</p>
	**********************************************************************/

	private String line_separator = null;



	/**********************************************************************
	Sets line separator.

	<p><b>Details:</b>  <code>setLineSeparator</code> sets the current line separator which <code>writeln</code> will use.  If <code>ls</code> is <code>null</code>, the platform's default line separator will be used.</p>

	@param ls the new line separator
	**********************************************************************/

	public synchronized void setLineSeparator (String ls)
	{
		line_separator = ls;
	}



	/**********************************************************************
	Returns current line separator.

	<p><b>Details:</b>  <code>getLineSeparator</code> returns the line separator currently being used by the <code>writeln</code> method.  Note that this value will never be <code>null</code>, even <code>null</code> was passed to <code>setLineSeparator</code>.</p>

	<p>The <code>writeln</code>s call this method once for each call, and use the result to terminate the line it is writing.  Thus, some interesting line termination effects may be achieved by subclassing and overriding this method.</p>

	@return the current line separator
	**********************************************************************/

	public synchronized String getLineSeparator ()
	{
		if (line_separator == null)
			line_separator = Platform.getLineSeparator ();
		return line_separator;
	}



	/**********************************************************************
	Writes ASCII string.

	<p><b>Details:</b>  write converts the given string (<var>text</var>) into an array of <code>byte</code>s and sends them to the target <code>OutputStream</code>.</p>

	<p>This method is called by <code>writeln</code>, so be careful when overriding it.</p>

	@param text the string to write
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public void write (String text) throws IOException
	{
		int length = text . length ();
		byte[] buff = new byte [length];
		for (int i = 0; i < length; ++ i)
			buff [i] = (byte) text . charAt (i);
		out . write (buff);
	}



	/**********************************************************************
	Writes ASCII line.

	<p><b>Details:</b>  <code>writeln</code> adds a line terminator to the given string (<var>line</var>) and forwards it to <code>write(String)</code>.</p>

	@param line the line to write
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public void writeln (String line) throws IOException
	{
		write (line + getLineSeparator ());
	}



	/**********************************************************************
	Terminates line.

	<p><b>Details:</b>  <code>writeln</code> forwards the current defined line terminator to <code>write(String)</code>.</p>

	@exception IOException if an I/O error occurs
	**********************************************************************/

	public void writeln () throws IOException
	{
		write (getLineSeparator ());
	}



}



