///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import lava.Platform;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;
import lava.string.WildcardExpression;



/******************************************************************************
Command-line file selection.

<p><b>Details:</b> <code>CommandLineFileSelection</code> is a powerful file-<wbr>selection parser designed for use in command-<wbr>line file processing utilities and is especially useful in programs that recursively process large trees of files.  <code>CommandLineFileSelection</code>'s file set selection algorithm is robust enough to provide powerful selection expressions and smart enough to recognize duplicate and redundant file listings in the selection.  When used within a Unix environment, symlinks are properly handled as well.</p>

<p>To effectively utilize <code>CommandLineFileSelection</code>, your command-<wbr>line program should first parse its command line arguments in its own application-<wbr>specific manner.  During parsing, your program should collect the <code>CommandLineFileSelection</code>-specific arguments into a new vector and pass them into <code>CommandLineFileSelection</code>'s constructor.  Then, your application can use <code>CommandLineFileSelection</code>'s enumerator to walk through the list of selected files.</p>

<p><b>file set selection syntax</b></p>

<p>The file set selection syntax recognized by CommandLineFileSelection is as follows:</p>

<blockquote>

	<p>
		&lt;<var>fs_argument_list</var>&gt; =&gt; &lt;<var>section</var>&gt; [&lt;<var>fs_argument_list</var>&gt;]
	</p>

	<p>
		&lt;<var>fs_section</var>&gt; =&gt; &lt;<var>include_paths</var>&gt;<br>
		&lt;<var>fs_section</var>&gt; =&gt; &lt;<var>exclude_paths</var>&gt;<br>
		&lt;<var>fs_section</var>&gt; =&gt; &lt;<var>include_masks</var>&gt;<br>
		&lt;<var>fs_section</var>&gt; =&gt; &lt;<var>exclude_masks</var>&gt;
	</p>

	<p>
		&lt;<var>include_paths</var>&gt; =&gt; <code>+p</code> &lt;<var>path_list</var>&gt;
	</p>

	<p>
		&lt;<var>exclude_paths</var>&gt; =&gt; <code>-p</code> &lt;<var>path_list</var>&gt;
	</p>

	<p>
		&lt;<var>include_masks</var>&gt; =&gt; <code>+m</code> &lt;<var>mask_list</var>&gt;
	</p>

	<p>
		&lt;<var>exclude_masks</var>&gt; =&gt; <code>-m</code> &lt;<var>mask_list</var>&gt;
	</p>

	<p>
		&lt;<var>path_list</var>&gt; =&gt; &lt;<var>path</var>&gt; [&lt;<var>path_list</var>&gt;]
	</p>

	<p>
		&lt;<var>mask_list</var>&gt; =&gt; &lt;<var>mask</var>&gt; [&lt;<var>mask_list</var>&gt;]
	</p>

</blockquote>

<p>A <var>path</var> is any string describing a valid path on the host platform.  A <var>mask</var> is any string of characters including '?' and '*', where '?' matches any character in the file name and '*' matches any substring in the file name.</p>

<p>The file selection criteria is as follows:  A single path indicates a file subset.  If the path is a file, then the subset has only that one file.  If the path is a directory, however, then every file in that directory, its subdirectories, and so on, is in the subset.  Each subset is marked for addition to or subtraction from the total set of files.  The file subsets are added and subtracted from the total set in the order they are given.  The resulting set is the list of files that are candidates for modification.</p>

<p>Next, the masks are processed.  Like the file subsets, masks are marked for inclusion or exclusion.  A positive accepting mask makes a file eligible.  A negative accepting mask disqualifies the file.  The masks are applied in order.  When two masks conflict, the mask occuring later in the list wins.  By default, all files are accepted if no masks are specified.  Otherwise, the last applicable mask must accept the file in order for the file to be included in the set.</p>

<p><b>useful tips</b></p>

<p>If your application requires the user to specify masks on the command line, be sure to recommend that they surround the masks by quotes to prevent the shell interpreter from automatically expanding them.</p>

<p>Because <code>CommandLineFileSelection</code> implements such a powerful and versatile file selection process, your program should give the user an opportunity to preview the selected file set before actually applying the program's file processing operation.</p>

@since 2001.01.26
@version 2001.03.09
@author Sharky
******************************************************************************/

public class CommandLineFileSelection
{



	/**********************************************************************
	Initializes file list.

	<p><b>Details:</b>  This constructor initializes a new <code>CommandLineFileSelection</code> with the provided argument list.</p>

	<p>The format for the argument list is given in the class documentation.</p>

	@param args the argument list
	@param first index of first argument

	@version 2001.03.02
	**********************************************************************/

	public CommandLineFileSelection (String[] args, int first) throws IOException
	{
		processArguments (args, first);
	}



	/**********************************************************************
	Initializes file list.

	<p><b>Details:</b>  This constructor initializes a new <var>CommandLineFileSelection</var> with the provided argument list.  Each element in the vector is assumed to be a String.</p>

	<p>The format for the argument list is given in the class documentation.</p>

	@param args the argument list
	**********************************************************************/

	public CommandLineFileSelection (Vector args) throws IOException
	{
		String[] array = new String [args . size ()];
		args . copyInto (array);
		processArguments (array, 0);
	}



	private Vector masks = new Vector ();



	private Vector mask_signs = new Vector ();



	private PathSet files = new PathSet (false);



	private void processArguments (String[] args, int start) throws IOException
	{
		String lastarg = null;
		for (int i = start; i < args . length; ++ i)
		{
			String arg = args [i];
			if
			(
				arg . equals ("+p")
			||	arg . equals ("-p")
			||	arg . equals ("+m")
			||	arg . equals ("-m")
			)
				lastarg = arg;
			else if (lastarg == null)
			{
				throw new IllegalArgumentException ("don't understand argument " + arg);
			}
			else if (lastarg . equals ("+m"))
			{
				masks . addElement (getWildcardExpression (arg));
				mask_signs . addElement (Boolean.TRUE);
			}
			else if (lastarg . equals ("-m"))
			{
				masks . addElement (getWildcardExpression (arg));
				mask_signs . addElement (Boolean.FALSE);
			}
			else if (lastarg . equals ("+p"))
				addFiles (arg);
			else if (lastarg . equals ("-p"))
				removeFiles (arg);
			else
				throw new lava.UnreachableCodeException ();
		}
	}



	private static final WildcardExpression getWildcardExpression (String s)
	{
		if (WINDOWS ())
			s = s . toUpperCase ();
		return new WildcardExpression (s);
	}



	private void addFiles (String path) throws IOException
	{
		files . addRecursively (new File (path));
	}



	private void removeFiles (String path) throws IOException
	{
		files . removeRecursively (new File (path));
	}



	public Enumeration getSelection ()
	{
		Vector passed_files = new Vector ();
		Enumeration enum = files . enumerate ();
		while (enum . hasMoreElements ())
		{
			Path p = (Path) enum . nextElement ();
			if (passesMask (p))
				passed_files . addElement (p);
		}
		return passed_files . elements ();
	}



	private boolean passesMask (Path file)
	{
		if (masks . size () == 0)
			return true;
		String filename = file . getFileName ();
		if (WINDOWS ())
			filename = filename . toUpperCase ();
		boolean pass = false;
		Enumeration e1 = masks . elements ();
		Enumeration e2 = mask_signs . elements ();
		while (e1 . hasMoreElements ())
		{
			WildcardExpression mask = (WildcardExpression) e1 . nextElement ();
			boolean mask_sign = ((Boolean) e2 . nextElement ()) . booleanValue ();
			if (mask . accept (filename))
				pass = mask_sign;
		}
		return pass;
	}



	private static boolean WINDOWS ()
	{
		return Platform.getOsGenre () == Platform.WINDOWS;
	}



}



