///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.IOException;
import java.util.StringTokenizer;



/******************************************************************************
Console-based command interface.

<p><b>Details:</b> A <code>ConsoleCommandShell</code> interacts with the user by issuing prompts and parsing the user's responses.  When instructed, this class writes a prompt to the console, waits for the user to type a line of text, and then begins parsing the line.  Each line is divided into substrings, which are returned one at a time during subsequent method calls.  By convention, the first substring is called the "command," while all additional substrings on the same line are called "parameters."</p>

<p>The algorithm used by this class to parse input lines is similar to the algorithms used in common command-line shells, such as in a Unix or MS-DOS shell.  For example:</p>

<ul>
	<li>Multiple words typed on a line are parsed into individual parameters.</li>
	<li>Multiple words are treated as a single parameter when the words are enclosed by quotes.</li>
	<li>Some characters in quoted strings are interpreted differently when preceded by a backslash.</li>
	<li>Long responses can be split into multiple lines by ending intermediate lines with a backslash.*</li>
</ul>

<blockquote><small>*This feature will be implemented in a future release.</small></blockquote>

<p>This class depends heavily on the {@link ConsoleDialog <code>ConsoleDialog</code>} class, particularly on {@link ConsoleDialog##query_words(java.lang.String, java.lang.String) <code>ConsoleDialog.query_words</code>}.  It is worthwhile to consult the documentation for that class in order to better understand the behavior of this class.</p>


@version 1999.05.04
@author Sharky
******************************************************************************/

public class ConsoleCommandShell
{



	/**********************************************************************
	Prompt sent to the user to indicate that a line of text (i.e., a command) is requested.
	**********************************************************************/

	public String prompt;



	/**********************************************************************
	Value assigned to word_buffer when the user enters a blank line.
	**********************************************************************/

	private static final String [] BLANK_LINE = new String [0];



	/**********************************************************************
	Words parsed from the current line.
	**********************************************************************/

	private String [] word_buffer = BLANK_LINE;



	/**********************************************************************
	Number of parameters read from the current line.  (This is an index into word_buffer.)
	**********************************************************************/

	private int words_read;



	private final ConsoleDialog console;



	/**********************************************************************
	Initializes a new <code>ConsoleCommandShell</code> with the specified prompt.

	@param the prompt
	**********************************************************************/

	public ConsoleCommandShell (ConsoleDialog console, String prompt)
	{
		this . console = console;
		this . prompt = prompt;
		word_buffer = null;
	}



	/**********************************************************************
	Outputs the command prompt to the console, waits for a response, parses it, and then returns the command (i.e., the first word of the response).

	@return the command, or null if a blank line was entered
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public String getCommand () throws IOException
	{
		words_read = 0;
		System . out . print (prompt);
		String [] new_buffer = console . queryWords ("", null);
		if (new_buffer == null)
			word_buffer = BLANK_LINE;
		else
			word_buffer = new_buffer;
		return getParam ();
	}



	/**********************************************************************
	Returns the next parameter from the most recently parsed command line.  This method should not be called before <code>get_command</code>.

	@return the next parameter, or null if there are no more words on the current line
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public String getParam () throws IOException
	{
		if (words_read >= word_buffer . length)
			return null;
		else
			return word_buffer [words_read ++];
	}



	/**********************************************************************
	Returns the number of parameters that remain to be read.

	@return number of parameters remaining to be read
	**********************************************************************/

	public int paramsRemaining ()
	{
		return word_buffer . length - words_read;
	}



	// added 1999.05.04
	public void writeln (String s) throws IOException
	{
		console . sendLine (s);
	}



}



