///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.util.Vector;
import lava.Platform;
import lava.clib.Ctype;
import lava.string.NumberString;
import java.io.BufferedInputStream;



/******************************************************************************
Console-based input manager.

<p><b>Details:</b> <code>ConsoleDialog</code> offers a set of methods which are useful for interacting with users in console mode applications.  Prompts are typically sent to the user through <code>System.out</code>, and responses are received from the user through <code>System.in</code>, but both streams can be customized.</p>

<p>The primary benefit of using <code>ConsoleDialog</code> is that it makes obtaining certain kinds of input easy.  For example, you can call a method that requests an integer from the user and know that the return value will be a valid integer, since it will not return until the user complies.  Deep in the call, that method automatically handles parsing, re-prompts after invalid responses, etc., so you don't have to.</p>

<p><b>query* methods</b></p>

<p>The <code>query</code>* methods prompt the user for information and then wait for a response.  If the user's response is valid, the method returns with the data.  If the user's response is not valid, then the action taken depends on the value of the <code>error_response</code> parameter passed in.</p>

<dl>
 <dt>
  <b>Case 1:</b>
  <code>error_response != null</code>
  <dd>
   The <code>error_response</code> string is output to the console and the user is prompted for the information again.  (Note that a newline character is <em>not</em> automatically appended to the error message.)  The method will never return until the user has entered a valid response.
 <dt>
  <b>Case 2:</b>
  <code>error_response == null</code>
  <dd>
   No error message is output.  Instead, a special value is returned to indicate failure.
 <dt>
  <b>Case 3:</b>
  <code>error_response</code> was omitted from the parameter call
  <dd>
   Same as Case 1, except that a default error message is used.
</dl>

<p>Note that in case (2), it may be difficult to discern valid responses from failure values if the failure values lie within the range of valid responses.  (See the documentation for each method for more information on which methods return which failure values.)  In cases (1) and (3), the methods do not return until the user has entered a valid response.</p>

<p>Spaces before and after user responses are allowed, but always ignored.  If the first part of a response is valid but the line contains additional input, the entire response is rejected.  If an EOF signal is detected while reading the user's response, an <code>EOFException</code> is thrown.</p>

@author Sharky
******************************************************************************/

final public class ConsoleDialog
{



	/*********************************************************************
	The input stream for responses.
	*********************************************************************/

	private final UnlimitedPushbackReader in;



	/*********************************************************************
	The output stream for prompts.
	*********************************************************************/

	private final Writer out;



	/**********************************************************************
	Constructs an instance that sends prompts to the specified output stream and reads responses from the specified input stream.

	@param reader the input stream
	@param writer the output stream
	**********************************************************************/

	public ConsoleDialog (Reader reader, Writer writer)
	{
		in = new UnlimitedPushbackReader (new UnixLineReader (new BufferedReader (reader)));
		out = writer;
	}



	/**********************************************************************
	Constructs an instance that sends prompts to <code>System.out</code> and receives responses from the specified <code>Reader</code>.
	**********************************************************************/

	public ConsoleDialog (Reader reader)
	{
		this
		(
			reader,
			new OutputStreamWriter (System.out)
		);
	}



	/**********************************************************************
	Constructs an instance that sends prompts to <code>System.out</code> and receives responses from <code>System.in</code>.
	**********************************************************************/

	public ConsoleDialog ()
	{
		this (new InputStreamReader (new BufferedInputStream (System.in, 1)));
	}



	/*********************************************************************
	Default response to be used when the error response parameter is omitted from a <code>query</code> method call.
	*********************************************************************/

	public String default_error_response =
		"invalid response" + LineSeparatorFilter.DEFAULT_SEPARATOR;



	/*********************************************************************
	Writes a string to the output stream and flushes.
	*********************************************************************/

	public void send (String s) throws IOException
	{
		out . write (s);
		out . flush ();
	}



	/*********************************************************************
	Writes a line to the output stream and flushes.
	@since 1998.11.12
	*********************************************************************/

	public void sendLine (String s) throws IOException
	{
		out . write (s);
		out . write (LineSeparatorFilter.DEFAULT_SEPARATOR);
		out . flush ();
	}



	/*********************************************************************
	Reads a line of input.</p>

	@return the line that was read, not including the line separator, or <code>null</code> if EOF
	*********************************************************************/

	public String readLine () throws IOException
	{
		return IoToolbox.readLine (in);
	}



	/*********************************************************************
	Queries the user for a <code>float</code>.  A valid response is any string of characters which can reasonably be interpreted as a <code>float</code>.  For more information about this method, see the general description of <code>query</code>* methods above.  <b>Failure value:</b> <code>Float.MIN_VALUE</code>.

	@param prompt the prompt
	@param error_response the error response (<b>default:</b> <code>default_error_response</code>)
	@return the float entered by the user
	*********************************************************************/

	public float queryFloat
	(
		String prompt,
		String error_response
	)
		throws IOException
	{
		while (true)
		{
			send (prompt);
			String s = readLine ();
			if (s == null)
				throw new EOFException ();
			s = s . trim ();
			Float f;
			try
			{
				f = Float.valueOf (s);
			}
			catch (NumberFormatException e)
			{
				f = null;
			}
			if (f != null)
				return f . floatValue ();
			if (error_response == null)
				return Float . MIN_VALUE;
			send (error_response);
		}
	}



	/*********************************************************************
	Queries the user for a <code>float</code>.  Supplies default parameters for <code>queryFloat(String,String)</code>.
	*********************************************************************/

	public float queryFloat (String prompt) throws IOException
	{
		return queryFloat (prompt, default_error_response);
	}



	/*********************************************************************
	Queries the user for an <code>int</code>.  A valid response is any string of characters which can be interpreted as a signed 32-bit integer using the given radix.  Radix-specific prefixes, such as "0x", are not allowed.  For more information about this method, see the general description of <code>query</code>* methods above.  <b>Failure value:</b> <code>Integer.MIN_VALUE</code>.

	@param prompt the prompt
	@param radix the radix (<b>default:</b> 10)
	@param error_response the error response (<b>default:</b> <code>default_error_response</code>)
	@return the int entered by the user
	*********************************************************************/

	public int queryInt
	(
		String prompt,
		int radix,
		String error_response
	)
		throws IOException
	{
		while (true)
		{
			send (prompt);
			String s = readLine ();
			if (s == null)
				throw new EOFException ();
			s = s . trim ();
			Integer i = NumberString.toSignedInt (s, radix);
			if (i != null)
				return i . intValue ();
			if (error_response == null)
				return Integer.MIN_VALUE;
			send (error_response);
		}
	}



	/*********************************************************************
	Queries the user for an <code>int</code>.  Supplies default parameters for <code>queryInt(String,int,String)</code>.
	*********************************************************************/

	public int queryInt (String prompt, int radix) throws IOException
	{
		return queryInt (prompt, radix, default_error_response);
	}



	/*********************************************************************
	Queries the user for an <code>int</code>.  Supplies default parameters for <code>queryInt(String,int,String)</code>.
	*********************************************************************/

	public int queryInt (String prompt) throws IOException
	{
		return queryInt (prompt, 10, default_error_response);
	}



	/*********************************************************************
	Queries the user for a list of words.  A word is defined as any continuous sequence of non-space characters, but spaces may be included in words by surrounding the words with double quotes.  Double-quoted strings are interpreted like C-style string literals.  Unpaired quotes are treated as literal quotes.  For more information about this method, see the general description of <code>query</code>* methods above.  If the line entered is blank, a 0-length array is returned.  (Note that a blank line is <em>not</em> considered a failure.)</p>

	<p><b>Failure value:</b> <code>null</code>.</p>

	@param prompt the prompt
	@param error_response the error response (<b>Default value:</b> <code>default_error_response</code>)
	@return an array of <code>String</code>s containing the words entered by the user
	*********************************************************************/

	public String [] queryWords
	(
		String prompt,
		String error_response
	)
		throws IOException
	{
	outer:
		while (true)
		{
			send (prompt);
			Vector vector = new Vector ();
			while (true)
			{
				StreamParser.tryHorizontalWhiteString (in);
				String s = StreamParser.tryShellArgument (in);
				if (s == null)
				{
					s = IoToolbox.readLine (in);
					if (s == null)
						throw new EOFException ();
					if (s . length () != 0)
					{
/* ENABLE THIS CODE TO REJECT STRINGS THAT ARE MISSING TERMINATING QUOTES
						if (error_response == null)
							return null;
						send (error_response);
						continue outer;
*/
						vector . addElement (s);
					}
					break;
				}
				vector . addElement (s);
			}
			int count = vector . size ();
			String[] words = new String [count];
			vector . copyInto (words);
			return words;
		}
	}



	/*********************************************************************
	Queries the user for a list of words.  Supplies default parameters for <code>queryWords(String,String)</code>.
	*********************************************************************/

	public String [] queryWords (String prompt) throws IOException
	{
		return queryWords (prompt, default_error_response);
	}



	/*********************************************************************
	Queries the user for a single word or C-style quoted string.  For more information about this method, see the general description of <code>query</code>* methods above.</p>

	<p><b>Failure value:</b> <code>null</code>.</p>

	@param prompt the prompt
	@param error_response the error response
	@return a String containing the word entered by the user
	*********************************************************************/

	public String queryWord
	(
		String prompt,
		String error_response
	)
		throws IOException
	{
		String [] words;
		while (true)
		{
			words = queryWords (prompt, null);
			if (words == null)
				return null;
			if (words . length == 1)
				break;
			if (error_response == null)
				return null;
			send (error_response);
		}
		return words [0];
	}



	/*********************************************************************
	Queries the user for a single word or C-style quoted string.  Supplies default parameters for <code>queryWord(String,String)</code>.
	*********************************************************************/

	public String queryWord (String prompt) throws IOException
	{
		return queryWord (prompt, default_error_response);
	}



	/*********************************************************************
	Queries the user for a single character.  For more information about this method, see the general description of <code>query</code>* methods above.</p>

	<p><b>Failure value:</b> <code>0</code>.</p>

	@param prompt the prompt
	@param error_response the error response
	@return the character entered by the user

	@since 1998
	@version 2000.06.03
	*********************************************************************/

	public char queryChar (String prompt, String error_response) throws IOException
	{
		while (true)
		{
			String line = IoToolbox.readLine (in);
			if (line == null)
				throw new EOFException ();
			line = line . trim ();
			if (line . length () == 1)
				return line . charAt (0);
			if (error_response != null)
				send (error_response);
			else
				return 0;
		}
	}



	/*********************************************************************
	Queries the user for a single character, using the default error response.  For more information about this method, see the general description of <code>query</code>* methods above.</p>

	@param prompt the prompt
	@return the character entered by the user
	*********************************************************************/

	public char queryChar (String prompt) throws IOException
	{
		return queryChar (prompt, default_error_response);
	}



	private final static boolean isMicrosoftPlatform;



	static
	{
		isMicrosoftPlatform = Platform.getOsGenre () == Platform.WINDOWS;
	}



	private static boolean containsDriveLetter (String name)
	{
		if
		(
			! isMicrosoftPlatform
		||
			name . length () < 2
		||
			! Ctype . isalpha (name . charAt (0))
		||
			name . charAt (1) != ':'
		)
			return false;
		return true;
	}



	private static File resolveFile (String dir, String name)
	{
		File file = new File (name);
		if (file . isAbsolute ())
			return file;
		if (containsDriveLetter (name))
			return file;
		return new File (dir, name);
	}



	/*********************************************************************
	Queries the user for an output filename.  A valid response is filename that describes an existing, readable file.  Filenames with spaces must be enclosed in quotes.  The user may response with a relative filename or an absolute filename.  If a relative filename is given, the <code>directory</code> parameter will be used to resolve the filename.</p>

	<p>For more information about this method, see the general description of <code>query</code>* methods above.</p>

	<p><b>Failure value:</b> <code>null</code>.</p>

	<p><b>Note:</b> The following boolean condition is tested to make sure the filename entered is a valid file:</p>

	<blockquote><code>
	file . exists () &amp;&amp; file . isFile () &amp;&amp; file . canRead ()
	</code></blockquote>

	where <code>file</code> is a <code>java.io.File</code> object.

	@param prompt the prompt
	@param directory the starting directory for relative filenames
	@param error_response the error response
	@return a <code>File</code> object refering to the file
	*********************************************************************/

	public File queryInputFile
	(
		String prompt,
		String directory,
		String error_response
	)
		throws IOException
	{
		while (true)
		{
			String name = queryWord (prompt);
			if (name == null)
				return null;
			File file = resolveFile (directory, name);
			if
			(
				file . exists ()
			&&	file . isFile ()
			&&	file . canRead ()
			)
				return file;
			if (error_response == null)
				return null;
			send (error_response);
		}
	}



	/*********************************************************************
	See <a href="#queryInputFile(java.lang.String, java.lang.String, java.lang.String)"><code>queryInputFile(String,String,String)</code></a>.
	*********************************************************************/

	public File queryInputFile (String prompt, String directory) throws IOException
	{
		return queryInputFile (prompt, directory, default_error_response);
	}



	/*********************************************************************
	Queries the user for an output filename.  A valid response is filename that describes a file that can be created or written to.  Filenames with spaces must be enclosed in quotes.  The user may response with a relative filename or an absolute filename.  If a relative filename is given, the <code>directory</code> parameter will be used to resolve the filename.</p>

	<p>For more information about this method, see the general description of <code>query</code>* methods above.</p>

	<p><b>Failure value:</b> <code>null</code>.</p>

	<p><b>Note:</b> The following boolean condition is tested to make sure the filename entered is a valid file:</p>

	<blockquote><code>
	file . canWrite ()
	</code></blockquote>

	where <code>file</code> is a <code>java.io.File</code> object.

	@param prompt the prompt
	@param directory the starting directory for relative filenames
	@param error_response the error response
	@return a <code>File</code> object refering to the selected output file
	*********************************************************************/

	public File queryOutputFile (String prompt, String directory, String error_response) throws IOException
	{
		while (true)
		{
			String name = queryWord (prompt);
			if (name == null)
				return null;
			File file = resolveFile (directory, name);
// Java bug requires workaround.
//			if (file . canWrite ())
//				return file;
			if (! file . exists () || file . canWrite ())
				return file;
			if (error_response == null)
				return null;
			send (error_response);
		}
	}



	/*********************************************************************
	See <a href="#queryOutputFile(java.lang.String, java.lang.String, java.lang.String)"><code>queryOutputFile(String,String,String)</code></a>.
	*********************************************************************/

	public File queryOutputFile (String prompt, String directory) throws IOException
	{
		return queryOutputFile (prompt, directory, default_error_response);
	}



}



