///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.io.InputStream;
import java.io.Writer;
import java.io.OutputStream;
import java.io.IOException;



/******************************************************************************
Rewrites files in place.

<p><b>Details:</b> A <code>FileRewriter</code> assists the user in rewriting files in place.  Normally, when an application rewrites a file while reading it, it does so as follows:  The file is accessed in read-only mode, while the revised contents are sent to a temporary file.  Then, after the entire source file has been processed, the source file is closed, deleted, and the temporary file is renamed to the location of the original file.  This class facilitates this process by shielding the user from most of these steps and performing them automatically.</p>

<p>Given a constructed instance of a <code>FileRewriter</code>, the user can obtain an input source by calling either <code>getInputStream</code> or <code>getReader</code>.  Note that only one of these two input sources can be selected for any given <code>FileRewriter</code> instance.  Similarly, an output destination can be obtained by calling either <code>getOutputStream</code> or <code>getWriter</code>, but not both.  The objects returned by these methods can then be used to read, filter, and rewrite the file's contents.</p>

<p>When you have finished rewriting the file, simply call the close method of the source stream and destination stream.  <strong>Be sure to close the source stream first, as closing the destination stream will cause the temporary file to automatically replace the source file.</strong></p>

<blockquote>
	<p><b>Example:</b> Rewrite the file 'story.txt', making sure all occurances of the letter 'a' are capitalized:</p>

	<xmp>
	*FileRewriter fr = new FileRewriter (new File ("story.txt"));
	*Reader reader = fr . getReader ();
	*Writer writer = fr . getWriter ();
	*while (true)
	*{
	*	int c = reader . read ();
	*	if (c < 0)
	*		break;
	*	if (c == 'a')
	*		writer . write ('A');
	*	else
	*		writer . write ((char) c);
	*}
	*reader . close();
	*writer . close (); // Closes and renames the dest file.
	</xmp>
</blockquote>

@author Sharky
******************************************************************************/

public class FileRewriter
{



	/**********************************************************************
	The file which is being rewritten.
	**********************************************************************/

	private File file;



	/**********************************************************************
	Initializes the members of this instance in preparation for rewriting the specified file.

	@param file the file to rewrite
	@exception FileNotFoundException if the specified file does not exist, is not readable, or is not writeable
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public FileRewriter (File file) throws FileNotFoundException, IOException
	{
		if (! file . exists ())
			throw new FileNotFoundException (file . toString ());
		if (! file . isFile ())
			throw new IOException ("can't rewrite " + file);
		if (! file . canWrite ())
			throw new IOException ("can't write to file");
		this . file = file;
	}



	/**********************************************************************
	The input source, either a Reader or InputStream.  null if no input source has been selected yet.
	**********************************************************************/

	private Object in = null;



	/**********************************************************************
	Returns an <code>InputStream</code> that reads from the file being rewritten.

	@return the input source for this <code>FileRewriter</code>
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public final InputStream getInputStream () throws IOException
	{
		if (in == null)
			in = new FileInputStream (file);
		return (InputStream) in;
	}



	/**********************************************************************
	Returns a <code>Reader</code> that reads from the file being rewritten.

	@return the input source for this <code>FileRewriter</code>
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public final Reader getReader () throws IOException
	{
		if (in == null)
			in = new FileReader (file);
		return (Reader) in;
	}



	/**********************************************************************
	The output source, either a Writer or OutputStream.  null if no output source has been selected.
	**********************************************************************/

	private Object out = null;



	/**********************************************************************
	Returns an <code>OutputStream</code> that writes to the file being rewritten.

	@return the output source for this <code>FileRewriter</code>
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public final OutputStream getOutputStream () throws IOException
	{
		if (out == null)
			out = new TemporaryFileOutputStream (file);
		return (OutputStream) out;
	}



	/**********************************************************************
	Returns a <code>Writer</code> that writes to the file being rewritten.

	@return the output source for this <code>FileRewriter</code>
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public final Writer getWriter () throws IOException
	{
		if (out == null)
			out = new TemporaryFileWriter (file);
		return (Writer) out;
	}



}



