///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.IOException;
import java.io.InputStream;
import java.io.FilterInputStream;



/******************************************************************************
Removes characters from base input stream.

<p><b>Details:</b>  <code>FilteredInputStream</code> is a <code>FilterInputStream</code> that accepts or rejects characters from the base input stream according to a specified criteria, on a per-character basis, passing on only accepted characters.  The acceptance criteria is determined by the implementation of the <code>accept</code> method.</p>

<p><b>Example:</b> The following program reads ASCII characters from <var>System.in</var> and outputs only the visible characters to <var>System.out</var>.  Note how the accept method is overridden directly in the constructor call.</p>

<form><textarea rows=25 cols=80 onChange="this.form.reset()";>
import java.io.IOException;
import lava.clib.Ctype;
import lava.io.FilteredInputStream;



class isgraph
{
*	public static void main (String[] args) throws IOException
*	{
*		FilteredInputStream in = new FilteredInputStream (System.in)
*		{
*			protected boolean accept (int c)
*			{
*				return Ctype.isgraph (c);
*			}
*		};
*		while (true)
*		{
*			int c = in . read ();
*			if (c < 0)
*				break;
*			System.out . print ((char) c);
*		}
*		in . close ();
*	}
}
</textarea></form>

@author Sharky

@version 1999.04.14
@since 1999.04.14
******************************************************************************/

public abstract class FilteredInputStream
	extends FilterInputStream
{



	/**********************************************************************
	Specifies base input stream.

	<p><b>Details:</b>  This constructor establishes <var>in</var> as the base input stream to be filtered.</p>

	@param in the base input stream
	**********************************************************************/

	public FilteredInputStream (InputStream in)
	{
		super (in);
		if (in == null)
			throw new NullPointerException ("in==null");
		this . in = in;
	}



	/**********************************************************************
	Returns a filtered character.

	<p><b>Details:</b>  <code>read</code> consumes a character from the base input stream, testing its acceptability by passing it to the <code>accept</code> method, and then returning the character if it has been accepted.  If the character is not accepted, <code>read</code> drops the character and tries the next character.  This continues until an acceptable character is found or the EOS is reached.  If the EOS is reached, read returns -1.</p>

	@return an acceptable character
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public int read () throws IOException
	{
		while (true)
		{
			int c = in . read ();
			if (c < 0)
				return -1;
			if (accept (c))
				return c;
		}
	}



	/**********************************************************************
	Defines acceptable characters.

	<p><b>Details:</b>  <code>accept</code> defines the behavior of this <code>FilteredInputStream</code> by determining whether characters should be accepted or rejected.  <code>accept</code> returns <code>true</code> if the supplied character is acceptable, <code>false</code> otherwise.</p>

	@param c the character
	@return true iff the character should be accepted
	**********************************************************************/

	protected abstract boolean accept (int c);



}



