///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.IOException;
import java.io.Reader;
import lava.clib.Ctype;



/******************************************************************************
Strips hashmark comments from a Reader.

<p><b>Details:</b> A HashCommentStripperReader filters incoming data from a <code>Reader</code> so that hash-marked comments are automatically omitted from the stream.  Comments begin with the first hash mark (#) and extend to the end of the line.  The end of the line is considered to be the first vertical space character, as determined by <code>lava.clib.Ctype.isvspace</code>.  Stripping can be disabled and reenabled on the fly.</p>

@version 1998.10.01
@since 1998.10.01
******************************************************************************/

public class HashCommentStripperReader
	extends LFilterReader
{



	/**********************************************************************
	Constructs a new instance that reads from the given <code>Reader</code>.

	@param reader the reader

	@version 1998.10.01
	@since 1998.10.01
	**********************************************************************/

	public HashCommentStripperReader (Reader reader)
	{
		super (reader);
	}



	/**********************************************************************
	Indicates whether stripping is currently enabled.

	<p><b>Details:</b> <var>stripping</var> indicates whether this <code>Reader</code> is currently stripping hash comments.  <var>stripping</var> is initially <code>true</code>.</p>

	@version 1998.10.01
	@since 1998.10.01
	**********************************************************************/

	protected boolean stripping = true;



	public synchronized void setStripping (boolean enabled)
	{
		stripping = enabled;
	}



	/**********************************************************************
	Reads a character from the comment-stripped stream.

	@return the character
	@exception IOException if an I/O error occurs

	@version 1998.10.01
	@since 1998.10.01
	**********************************************************************/

	public int read () throws IOException
	{
		return _read ();
	}



	/**********************************************************************
	Reads a character from the comment-stripped stream.

	@return the character
	@exception IOException if an I/O error occurs

	@version 1998.10.01
	@since 1998.10.01
	**********************************************************************/

	final int _read () throws IOException
	{
		boolean in_comment = false;
		int c;
		while (true)
		{
			c = in . read ();
			if (! stripping || c < 0)
				break;
			if (in_comment)
			{
				if (Ctype.isvspace (c))
					break;
			}
			else
			{
				if (c != '#')
					break;
				in_comment = true;
			}
		}
		return c;
	}



	/**********************************************************************
	Reads a block of characters from the comment-stripped stream.

	@param dest the destination array
	@param off offset into dest
	@param len number of characters to read
	@param the number of characters actually read
	@exception IOException if an I/O error occurs

	@version 2000.11.06
	@since 1998.10.01
	**********************************************************************/

	public int read (char [] dest, int off, int len) throws IOException
	{
		int i;
		for (i = 0; i < len; ++ i)
		{
			int c = _read ();
			if (c < 0)
				break;
			dest [off + i] = (char) c;
		}
		if (i == 0)
			i = len <= 0 ? 0 : -1;
		return i;
	}



}



