///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.PushbackReader;
import java.io.Reader;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import lava.clib.Ctype;
import lava.io.HashCommentStripperReader;



/******************************************************************************
Miscellaneous I/O functions.

<p><b>Details:</b> This class contains a variety of miscellaneous I/O-related utility functions and definitions.</p>

<p><b>Changes:</b></p>

<dl>
	<dt>2000.12.18
		<dd>Added preloadStream (Reader, int).
</dl>

@version 2000.12.18
@since 1998.10.11

@author Sharky
******************************************************************************/

public final class IoToolbox
{



	/**********************************************************************
	Computes a message digest from a stream.

	<p><b>Details:</b>  <code>computeStreamDigest</code> computes a message digest for the supplied input stream, closes the stream, and then returns the digest.</p>

	<p>The digest name must be one of the standard digest names provided in the Java Security API.</p>

	@param is the input stream
	@param algo the digest algorithm
	@return the digest
	@exception IOException if an I/O error occurs

	@since 1999.10.21
	**********************************************************************/

	public static byte[] computeStreamDigest (InputStream is, String algo)
		throws IOException
	{
		try
		{
			MessageDigest md;
			try
			{
				md = MessageDigest . getInstance (algo);
			}
			catch (NoSuchAlgorithmException e)
			{
				throw new IllegalArgumentException (algo);
			}
			if (! (is instanceof BufferedInputStream))
				is = new BufferedInputStream (is);
			byte [] buff = new byte [1024];
			while (true)
			{
				int n = is . read (buff);
				if (n < 0)
					break;
				md . update (buff, 0, n);
			}
			return md . digest ();
		}
		finally
		{
			IoCloser.close (is);
		}
	}



	/**********************************************************************
	Pushes string onto PushbackReader.

	<p><b>Details:</b>  unread pushes the given string (s) onto the given PushbackReader (in).</p>

	@param in the PushbackReader
	@param s the string
	@exception IOException if an I/O error occurs

	@since 2000.06.01
	**********************************************************************/

	public static void unread (PushbackReader in, String s) throws IOException
	{
		in . unread (s . toCharArray ());
	}



	//2000.06.01
	public static int peek (PushbackReader in) throws IOException
	{
		int c = in . read ();
		if (c >= 0)
			in . unread (c);
		return c;
	}



	//2000.06.03
	public static String readLine (Reader r) throws IOException
	{
		StringBuffer line = new StringBuffer ();
		while (true)
		{
			int c = r . read ();
			if (c < 0)
				if (line . length () > 0)
					break;
				else
					return null;
			if (Ctype.isvspace (c))
				break;
			line . append ((char) c);
		}
		return line . toString ();
	}



	/**********************************************************************
	Prints lines.

	<p><b>Details:</b>  printLines calls <code>System.out . println()</code> on each of the strings in the given array (<var>lines</var>).  (For example, you can use this to print a program usage summary when your command line program is run without arguments.)</p>

	@param lines lines to print

	@since 2000.09.24
	**********************************************************************/

	//2000.09.24
	public static void printLines (String[] lines)
	{
		final int length = lines . length;
		for (int i = 0; i < length; ++ i)
			System.out . println (lines [i]);
	}



	/**********************************************************************
	Returns reader suitable for parsing config files.

	<p><b>Details:</b>  <code>getConfigFileReader</code> returns a LineNumberReader that reads from configuration file whose path is (<var>name</var>).  The returned reader automatically trims away Unix-style hashed comments.</p>

	@param name path to config file
	@return config file reader
	@exception IOException if an I/O error occurs

	@since 2000.11.08
	**********************************************************************/

	public static LineNumberReader getConfigFileReader (String name) throws IOException
	{
		return new LineNumberReader
		(
			new BufferedReader
			(
				new HashCommentStripperReader
				(
					new FileReader (name)
				)
			)
		);
	}



	/**********************************************************************
	Creates reader for stdin.

	<p><b>Details:</b>  <code>getStdinReader</code> returns a <code>BufferedReader</code> attached to <code>System.in</code>.</p>

	@return the reader
	@exception IOException if an I/O error occurs

	@since 2000.11.08
	**********************************************************************/

	public static BufferedReader getStdinReader () throws IOException
	{
		return new BufferedReader (new InputStreamReader (System.in));
	}



	/**********************************************************************
	Loads entire stream into string.

	<p><b>Details:</b>  preloadStream loads the entire given stream (<var>reader</var>) into a string and returns it.  The <var>size</var> parameter offers a hint to this implementation about the total length of the stream, and may be set to 0 if the length of the stream is not known.</p>

	@param reader the stream
	@param size estimated buffer size needed
	@return the string containing the contents of the stream
	@exception IOException if an I/O error occurs

	@since 2000.12.19
	**********************************************************************/

	public static String preloadStream (Reader reader, int size) throws IOException
	{
		StringBuffer buff = new StringBuffer (size);
		while (true)
		{
			int c = reader . read ();
			if (c < 0)
				break;
			buff . append ((char) c);
		}
		reader . close ();
		return buff . toString ();
	}



}



