///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.BufferedReader;
import java.io.IOException;
import java.util.Enumeration;
import java.util.NoSuchElementException;



/******************************************************************************
Converts a Reader into an Enumeration of lines.

<p><b>Details:</b>  <code>LineEnumeration</code> is a wrapper class for <code>BufferedReader</code> that converts the <code>Reader</code>'s lines into a standard <code>Enumeration</code>.  Each line is served as one <code>Object</code> in the <code>Enumeration</code>.  An instance of this class can be supplied to any method in any other class that expects an <code>Enumeration</code>.  The <code>Enumeration</code> continues until the <code>Reader</code>'s EOS (end of stream) has been reach.  When the EOS is reached, <code>LineEnumeration</code> automatically closes the underlying stream.</p>

@see lava.util.Enumeration
@see lava.io.BufferedReader

@version 1998.11.05
@since 1998.11.05
******************************************************************************/

public class LineEnumeration
	implements Enumeration
{



	/**********************************************************************
	Base stream.

	<p><b>Details:</b>  <code>in</code> is the base <code>Reader</code> from which this Enumeration is derived.</p>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	private final BufferedReader in;



	/**********************************************************************
	Specifies the base Reader.

	<p><b>Details:</b>  This constructor initializes a new instance with <code>in</code> as the base Reader.</p>

	@param in the base stream

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	public LineEnumeration (BufferedReader in)
	{
		this . in = in;
	}



	/**********************************************************************
	Next line in the enumeration.

	<p><b>Details:</b>  <code>next_line</code> is the next line in the <code>Enumeration</code>.  This value will be returned when <code>nextElement</code> or <code>nextLine</code> is called.</p>

	<p>Often, methods in this class must peek ahead at the next line in the base <code>Reader</code>.  Rather than pushing it back onto the stream, it is temporarily stored here instead.  <code>next_line</code> may be <code>null</code>, indicating that the next line hasn't been peeked yet.</p>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	private String next_line = null;



	/**********************************************************************
	Prefetches the next line.

	<p><b>Details:</b>  If <code>next_line</code> is non-<code>null</code>, <code>buffer_line</code> does nothing.  Otherwise, it attempts to read a line from the base <code>Reader</code> and store it in <code>next_line</code>.  If an <code>IOException</code> occurs while reading, the exception is caught, but the call is aborted, and the base <code>Reader</code> is closed.</p>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	private void buffer_line ()
	{
		if (next_line != null)
			return;
		try
		{
			next_line = in . readLine ();
		}
		catch (IOException e)
		{
			ioe = e;
		}
		if (next_line == null)
			IoCloser . close (in);
	}



	/**********************************************************************
	Indicates whether this <code>Enumeration</code> has more elements.

	<p><b>Details:</b>  <code>hasMoreLines</code> returns <code>true</code> if there is at least one more line to consume from this <code>Enumeration</code>, <code>false</code> otherwise.</p>

	@return <code>true</code> iff there are more lines to consume

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	public boolean hasMoreLines ()
	{
		buffer_line ();
		return next_line != null;
	}



	/**********************************************************************
	Returns the next line.

	<p><b>Details:</b>  <code>nextLine</code> fetches and returns the next line from this <code>Enumeration</code>.  If there are no more lines, <code>nextLine</code> returns <code>null</code>.</p>

	@return the next line

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	public String nextLine ()
	{
		buffer_line ();
		String line = next_line;
		next_line = null;
		return line;
	}



	/**********************************************************************
	Synonym for <code>hasMoreLines</code>.

	<p><b>Details:</b>  <code>hasMoreElements</code> is a synonym for <code>hasMoreLines</code>.  This synonym is included to satisfy the <code>Enumeration</code> interface.</p>

	@return return value of <code>hasMoreLines</code>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	public boolean hasMoreElements ()
	{
		return hasMoreLines ();
	}



	/**********************************************************************
	Returns the next element.

	<p><b>Details:</b>  <code>nextElement</code> is identical in function to <code>nextLine</code>, with the exception that if there is no next line, it throws a <code>NoSuchElementException</code> instead of returning <code>null</code>.  The returned <code>Object</code> is guaranteed to be a <code>String</code>.</p>

	@return the next line

	@version 1998.11.05
	@since 1998.12.12
	**********************************************************************/

	public Object nextElement ()
	{
		Object obj = nextLine ();
		if (obj == null)
			throw new NoSuchElementException ();
		return obj;
	}



	/**********************************************************************
	Last <code>IOException</code> caught.

	<p><b>Details:</b>  <code>ioe</code> is the last exception caught while reading from the base <code>Reader</code>.  Normally, this value should be <code>null</code> all the time, indicating that no exception has been thrown.</p>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	private IOException ioe = null;



	/**********************************************************************
	Returns <code>ioe</code>.

	<p><b>Details:</b>  <code>getError</code> returns <code>ioe</code>.</p>

	@return value of <code>ioe</code>

	@version 1998.11.05
	@since 1998.11.05
	**********************************************************************/

	public IOException getError ()
	{
		return ioe;
	}



}



