///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



/******************************************************************************
Filters line-separators from text stream.

<p><b>Details:</b> <code>LineSeparatorFilter</code> assists in the conversion of text streams between heterogeneous platforms.  The main difference between text streams created on different platforms is the code used to separate lines.  Windows, Unix, and Macintosh all subscribe to unique formats.  This class will take a stream of characters generated on any of these platforms, or even a mixed stream generated from multiple platforms, and produce <em>predictable</em> output that conforms to a single platform.</p>

@see UnixLineReader
@author Sharky
******************************************************************************/


final public class LineSeparatorFilter
	implements Cloneable
{



	private static final int NO_CHAR = 0x80000000;



	public static final String DEFAULT_SEPARATOR = System . getProperty ("line.separator");



	/**********************************************************************
	The Unix-style line separator ("\n").
	**********************************************************************/

	public static final String UNIX_LINE_SEPARATOR = "\n";



	/**********************************************************************
	The DOS/Windows-style line separator ("\r\n").
	**********************************************************************/

	public static final String DOS_LINE_SEPARATOR = "\r\n";



	/**********************************************************************
	The Macintosh-style line separator ("\r").
	**********************************************************************/

	public static final String MAC_LINE_SEPARATOR = "\r";



	private String separator;



	/**********************************************************************
	Initializes a new LineSeparatorFilter to replace all line separators with the system's platform-specific line separator.
	**********************************************************************/

	public LineSeparatorFilter ()
	{
		separator = DEFAULT_SEPARATOR;
	}



	/**********************************************************************
	Initializes a new LineSeparatorFilter to replace all line separators with the specified string.

	@param separator the line separator that will replace all others
	**********************************************************************/

	public LineSeparatorFilter (String separator)
	{
		this . separator = separator;
	}



	/**********************************************************************
	The previous character read, if it was a line separator character, or zero if it was not.
	**********************************************************************/

	private char previous_endline_char = 0;



	/**********************************************************************
	Total number of substitutions made so far.
	**********************************************************************/

	int subst_count = 0;



	/**********************************************************************
	Does the work of <code>filter(int)</code>.  This method is installed to prevent unexpected polymorphism problems.
	**********************************************************************/

	private int _filter (int c)
	{
		switch (c)
		{
		case '\r':
			if (previous_endline_char == '\n')
			{
				previous_endline_char = 0;
				return NO_CHAR;
			}
			else
			{
				previous_endline_char = '\r';
				++ subst_count;
				return '\n';
			}
		case '\n':
			if (previous_endline_char == '\r')
			{
				previous_endline_char = 0;
				return NO_CHAR;
			}
			else
			{
				previous_endline_char = '\n';
				++ subst_count;
				return '\n';
			}
		default:
			previous_endline_char = 0;
			return c;
		}
	}



	private StringBuffer _filter (char[] src)
	{
		int length = src . length;
		StringBuffer buff = new StringBuffer (length);
		for (int s = 0; s < length; ++ s)
		{
			int c = _filter (src [s]);
			switch (c)
			{
			case NO_CHAR:
				break;
			case '\n':
				buff . append (separator);
				break;
			default:
				buff . append ((char) c);
			}
		}
		return buff;
	}



	/**********************************************************************
	Returns a new array with the line separators replaced by the preferred line separator declared when this instance was created.

	@param src the source array
	@return a new array with converted line separators
	**********************************************************************/

	public char[] filter (char [] src)
	{
		return _filter (src) . toString () . toCharArray ();
	}



	/**********************************************************************
	Returns a new String with the line separators replaced by the preferred line separator declared when this instance was created.

	@param src the source String
	@return a new String with converted line separators
	**********************************************************************/

	public String filter (String src)
	{
		return _filter (src . toCharArray ()) . toString ();
	}



	/**********************************************************************
	Creates an independent copy of this instance.  Readers that use this class and support the mark operation may use this method to make a "backup copy" of this instance, in order to preserve its state at mark time.</p>

	<p><small><b>Note:</b> Although this method is declared to throw a CloneNotSupportedException, rest assured that it will never happen (unless this class is subclassed and the subclass does not support cloning).  The rules of Java inheritance required the useless declaration.  Sorry. :-(</small>

	@return the copy
	@exception CloneNotSupportedException if this instance cannot be cloned
	**********************************************************************/

	public Object clone () throws CloneNotSupportedException
	{
		return super . clone ();
	}



}



