///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.Reader;
import java.io.FilterReader;
import java.io.IOException;



/******************************************************************************
Adds mark support to Reader.

<p><b>Details:</b> Currently, the only <code>java.io</code> classes that support the <code>mark</code>, <code>reset</code>, and <code>readLine</code> methods are <code>BufferedReader</code> and its descendants.  <code>MarkReader</code> is designed to give you a choice -- to enable you to use mark and reset without requiring you to use a <code>BufferedReader</code>.  MarkReader allows you to add mark and reset capabilities to <em>any</em> <code>Reader</code>.</p>

<blockquote>
	<p><b>Example:</b> Suppose you want to read user input from the console, and occasionally mark the stream while doing so (so that you can go back and reread some portions).  With the current <code>java.io</code> package, the only way you can do this is to create a <code>BufferedReader</code>.</p>

	<p>However, if you do this, blocking may occur where blocking <em>shouldn't</em> occur.  This is because <code>BufferedReader</code> tries to completely refill the read buffer -- even, for example, if a carriage return has already been encountered midway through the fill.  Imagine the user's frustration when he is prompted to enter a line of text:  He enters it, pushes return, and sees that nothing will happen until he enters additional lines of text -- text for which he has not yet received the prompts -- just so that he can fill the buffer enough to make the call to <code>readLine</code> return!</p>

	<p>This serious problem has been reported as a bug in Java 1.1, and this class is a perfect workaround for it.  Now you can use <code>mark</code> and still avoid the <code>BufferedReader</code> bug that causes this problem.</p>
</blockquote>
******************************************************************************/

public final class MarkReader
	extends LFilterReader
{



	/**********************************************************************
	Returns a Reader, based on the supplied reader, that is guaranteed to have mark capabilities.  If the supplied reader supports mark already, it will simply be returned.  Otherwise, the supplied reader will be embedded into a new instance of <code>MarkReader</code>, and the new <code>MarkReader</code> will be returned.

	@param reader the reader to adapt
	@return a read that supports mark
	**********************************************************************/

	public static Reader adapt (Reader reader)
	{
		if (reader . markSupported ())
			return reader;
		return new MarkReader (reader);
	}



	/**********************************************************************
	Initializes a new <code>MarkReader</code> to read from the indicated Reader.

	@param reader the reader that this MarkReader will read from
	**********************************************************************/

	public MarkReader (Reader reader)
	{
		super (reader);
	}



	/**********************************************************************
	The current mark buffer.
	**********************************************************************/

	private char [] buffer = null;



	/**********************************************************************
	Index of the next character to read from the mark buffer.  Equals <code>size</code> if characters should <em>not</em> be read from the mark buffer.
	**********************************************************************/

	private int pos;



	/**********************************************************************
	The number of valid characters currently occupying the mark buffer.
	**********************************************************************/

	private int size;



	/**********************************************************************
	Reads the next character from the stream.

	@return the character read
	**********************************************************************/

	public int read () throws IOException
	{
		return _read ();
	}



	/**********************************************************************
	Does the work for read().  This method is installed to prevent unexpected polymorphism problems.
	**********************************************************************/

	final int _read () throws IOException
	{
		if (buffer == null)
			return in . read ();
		if (pos < size)
			return buffer [pos ++];
		int c = in . read ();
		if (c >= 0)
			if (pos == buffer . length)
				buffer = null;
			else
			{
				buffer [size ++] = (char) c;
				++ pos;
			}
		return c;
	}



	/**********************************************************************
	Reads a block of characters from the stream.

	@param dest destination array for storing characters read
	@param off starting offset into the destination array
	@param len number of characters to read
	@return number of characters actually read (may be lest than len, indicating EOF or IO trouble)
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public int read (char [] dest, int off, int len) throws IOException
	{
		return read_by_singles (dest, off, len);
	}



	private int read_by_singles (char [] dest, int off, int len) throws IOException
	{
		int i;
		for (i = 0; i < len; ++ i)
		{
			int c = _read ();
			if (c < 0)
				break;
			dest [off + i] = (char) c;
		}
		if (i == 0)
			i = len <= 0 ? 0 : -1;
		return i;
	}



	/**********************************************************************
	Mark the present position in this filtered stream.  Subsequent calls to <code>reset()</code> will attempt to reposition the stream to state it was at when this method was called.  There is a limit on how many characters can be read ahead before attempts to reset are no longer guaranteed to succeed.

	@param read_ahead_limit the read ahead limit
	**********************************************************************/

	public void mark (int read_ahead_limit) throws IOException
	{
		char [] new_buffer = new char [read_ahead_limit];
		int new_size = 0;
		if (buffer != null)
		{
			new_size = size - pos;
			System . arraycopy (buffer, pos, new_buffer, 0, new_size);
		}
		buffer = new_buffer;
		pos = 0;
		size = new_size;
	}



	/**********************************************************************
	Attempts to reset the stream.  If the stream has been marked, then this method will attempt to store the state of the stream at the time it was marked.  If the stream has not been marked, then this method will attempt to reset the stream in some way that is appropriate to this particular stream.

	@exception IOException if the stream cannot be reset
	**********************************************************************/

	public void reset () throws IOException
	{
		if (buffer == null)
			in . reset ();
		pos = 0;
	}



	/**********************************************************************
	Indicates that this Reader supports marking.

	@return <code>true</code>
	**********************************************************************/

	public boolean markSupported ()
	{
		return true;
	}



}


