///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;
import lava.Platform;
import lava.string.PathToolbox;
import lava.string.StringToolbox;



/******************************************************************************
File in virtual/physical Unix file system.

<p><b>Details:</b>  <code>Path</code> is a replacement for <code>java.io.File</code>, the class that Sun screwed up.  Unlike <code>File</code>, this class takes a hard stand on what paths should look like and how they are handled.  This allows you, the developer, to engage in worry-<wbr>free path manipulation without fear of platform-<wbr>dependent gotchas.  <code>Path</code> also reduces the developer's dependecy on <code>java.io.File</code>, which has always been plagued with quirky behavior.</p>

<h2>platform independence</h2>

<p><code>Path</code>'s most significant feature is that it provides a uniform, hassle-free interface for manipulating file paths.  This is achieved by <em>defining</em> the path syntax to work in a manner that does not require the developer be overly concerned with platform-<wbr>independence.  (I guess I already said this, didn't I?)  When using <code>Path</code>, for example, the developer can always assume that the path separator is "/", regardless of the platform, thereby liberating the programmer from the need to make silly calls such as <code>System.getProperty ("file.separator")</code> just to figure out how to build a path name.  Path detects the host system's needs and translates the paths for you when platform-<wbr>dependent representations are needed.</p>

<h2>path equivalence</h2>

<p>Unlike <code>File</code>, <code>Path</code> correctly detects when two file specifications are equivalent.  This makes it easy to detect duplicate files when large lists of files are being processed.</p>

<p>All path specifications are automatically reduced to their simplest form before they are interpreted or used by any other method.  For example, the path "bob/../anne/index.html" is automatically reduced to "anne/index.html" before it is processed, and to Path instances created with either path in the constructor would be equivalent to each other.</p>

<p>Also, the parent of the root directory is the root directory.  Thus, the paths "/", "/..", and "/../.." are all equivalent.</p>

<h2>platform mapping</h2>

<p>Path uses the UNIX style path naming conventions for all platforms.  Paths can be constructed from <code>File</code> objects.  When this happens, <code>Path</code> attempts to preserve the relativeness or absoluteness of the file being represented.</p>

<p>Because <code>Path</code> attempts to make paths on all file systems look like UNIX paths, the path names for non-<wbr>Unix platforms such as Windows and Macintosh have to be "unixized."  This is achieved as follows:</p>

<h3>Windows</h3>

<p><code>Path</code> presents a virtual file system that has the feel of a Unix file system but encapsulates all of the features of Window-<wbr>specific paths.</p>

<ul>
 <li>
  <b>Absolute paths:</b> In Windows, there are two kinds of absolute paths: paths to files on mounted (local or mapped) drives, and network paths to shared files on other machines.  Path converts these path names as follows:
  <ul>
   <li>
    <b>Mounted drives:</b> The unixized path begins with a '/', is followed by an upper-case drive letter, and then the rest of the path.  For example, the file "C:\WINDOWS\WIN.INI" becomes "/C/WINDOWS/WIN.INI".
   <li>
    <b>Network shares:</b> The unixized path begins with '/NET/', then the computer name, the share name, and finally the path to the file within the share.  For example, the network path "\\computername\sharename\dir1\dir2\file" is converted to "/NET/computername/sharename/dir1/dir2/file".
  </ul>
 <li>
  <b>Relative paths:</b>  In Windows, relative paths are somewhat complicated.  This is because there can be more than one drive, and each drive can have its own "current" working directory.  Thus, there are relative paths and semi-<wbr>relative paths.  <code>Path</code> represents these as follows:
  <ul>
   <li>
    <b>Relative paths:</b> Relative paths which don't contain a drive letter are truly relative because they are resolved from the current working directory on the currently active drive.  Path easily represents these paths by converting the backslashed characters into forward slash characters.  Thus, the Windows relative path "media\button.gif" is simply "media/button.gif".
   <li>
    <b>Semi-<wbr>relative paths:</b> Relative paths which contain a drive letter are semi-<wbr>relative, because the drive letter portion is absolute while the actual path portion is relative.  The relative portion is resolved from the current working directory on the absolutely-<wbr>specified drive.  <code>Path</code> represents DOS-<wbr>based semi-<wbr>relative path specifications by converting them to absolute paths first and treating in the manner described above for absolute paths.  For example, if the current directory on the A: drive was "A:\data" and the semi-<wbr>relative path name "a:settings.dat" was used in a Path constructor, the path represented would be "/A/data/settings.dat".
  </ul>
</ul>

<p>Since <code>Path</code> is pure Java, <code>Path</code> internally uses <code>java.io.File</code> for many of its operations.  This often requires the creation of a <code>File</code> object from a <code>Path</code> object.  In many instances, the conversion may be impossible.  For example, there is no straightforward way to convert the path "/etc/passwd" into a Windows <code>File</code>, since this path does not include a representation of a drive letter.  On Windows platforms, Path automatically treats non-drive-specifying absolute paths as absolute paths on the current working drive.  Care should be taken when using this feature if the current working drive is subject to change within your Windows-Java application.</p>

<h4>case sensitivity</h4>

<p>Unix is case-<wbr>sensitive; Windows is not.  Therefore, to allow the Windows file system to behave more like a Unix file system, paths matching filenames whose cases are different are considered to be distinct.  For example, if the Window file C:\AUTOEXEC.BAT exists, then the virtual Unix file /C/AUTOEXEC.BAT exists, but /C/autoexec.bat does not.</p>

<h3>Macintosh</h3>

<p>Path does not yet support Macintosh platforms due to lack of information about Macintosh pathname formats and how they are interpreted by the <code>java.io.File</code> object.  Help would be appreciated.</p>

@version 2000.11.11
@author Sharky
******************************************************************************/

public class Path
	implements Comparable
{



	/**********************************************************************
	Path name.

	<p><b>Details:</b>  <var>pathname</var> is the name of the path represented by this instance.  <var>pathname</var> may or may not begin with a '/', depending on whether this <code>Path</code> represents an absolute path or a relative path.  Except for "/", which is the name of the root path, no path's name actually ends with a slash '/'.  Even if a slash-<wbr>terminated path name is supplied to any of this class' constructors, the trailing slash is automatically trimmed.</p>
	**********************************************************************/

	private final String pathname;



	//////////////////////////////////
	//                              //
	//  constructors and factories  //
	//                              //
	//////////////////////////////////



	/**********************************************************************
	Initializes with relative or absolute path.

	<p><b>Details:</b>  This constructor initializes a new <code>Path</code> by setting the path equal to <var>pathname</var>.</p>

	@param pathname the path name
	**********************************************************************/

	public Path (String pathname)
	{
		this . pathname = PathToolbox.simplifyPath (pathname);
	}



	/**********************************************************************
	Resolve relative path.

	<p><b>Details:</b>  <code>CONDITIONAL_RELATIVE</code> indicates that the constructor should attempt to determine an path based on the given working directory (<var>base</var>) and the given relative path (<var>rel</var>).  If <var>rel</var> is an absolute path name (i.e., begins with '/'), then the working directory is ignored.  If both <var>base</var> and <var>rel</var> are relative, the resulting <code>Path</code> will also be relative.</p>

	<p>This construction mode differs from <code>FORCE_RELATIVE</code> in that the resulting path will only be formed from both strings if <var>rel</var> is a relative path.</p>
	**********************************************************************/

	public static final int CONDITIONAL_RELATIVE = 1;



	/**********************************************************************
	Concatenates two paths.

	<p><b>Details:</b>  <code>FORCE_RELATIVE</code> indicates that the path should be formed by concatenating the two paths (<var>base</var>, <var>rel</var>), regardless of whether either is absolute or relative.  The result will be relative only if <var>base</var> is relative.</p>

	<p>Regardless of whether <var>base</var> ends with a slash or <var>rel</var> begins with a slash, or both, or neither, the constructor will make sure that exactly one slash appears between both components.</p>
	**********************************************************************/

	public static final int FORCE_RELATIVE = 2;



	/**********************************************************************
	Forces root dir on relative path.

	<p><b>Details:</b>  <code>CHANGE_ROOT</code> indicates that the path should be formed by treating <var>rel</var> as an absolute path, but with its root at <var>base</var>.  In particular, this means that even if <var>rel</var> contains many ".." components, the path will be shallower than <var>base</var>.</p>

	@since 2000.11.11
	**********************************************************************/

	public static final int CHANGE_ROOT = 3;



	/**********************************************************************
	Combines two path components.

	<p><b>Details:</b>  This constructor initializes a new <code>Path</code> based on two given path elements (<var>base</var>, <var>rel</var>).  The manner in which these elements are combined is determined by the given combination mode (<var>mode</var>).  Valid modes are <code>CONDITIONAL_RELATIVE</code> and <code>FORCE_RELATIVE</code>.  See the documentation for those identifiers for more information.</p>

	@param base the base component
	@param rel the relative component

	@see #CONDITIONAL_RELATIVE
	@see #FORCE_RELATIVE
	@see #CHANGE_ROOT
	**********************************************************************/

	public Path (String base, String rel, int mode)
	{
		String _pathname = null;
		switch (mode)
		{
		case CONDITIONAL_RELATIVE:
			_pathname = PathToolbox.resolveRelativePath (base, rel);
			break;
		case FORCE_RELATIVE:
			_pathname = PathToolbox.concatenatePaths (base, rel);
			break;
		case CHANGE_ROOT:
			_pathname = PathToolbox.concatenatePaths
			(
				base,
				PathToolbox.simplifyPath ("/" + rel)
			);
			break;
		default:
			throw new IllegalArgumentException ("mode=" + mode);
		}
		pathname = PathToolbox.simplifyPath (_pathname);
	}



	/**********************************************************************
	Combines two path components.

	<p><b>Details:</b>  This constructor does the same thing as <code>Path(String,String)</code>, except that the first parameter is given as an already constructed <code>Path</code>.</p>
	**********************************************************************/

	public Path (Path base, String rel, int mode)
	{
		this (base . pathname, rel, mode);
	}



	/**********************************************************************
	Combines two path components.

	<p><b>Details:</b>  This constructor does the same thing as <code>Path(String,String)</code>, except that the first two parameters are given as already constructed <code>Path</code>s.</p>
	**********************************************************************/

	public Path (Path base, Path rel, int mode)
	{
		this (base . pathname, rel . pathname, mode);
	}



	/**********************************************************************
	Returns working directory.

	<p><b>Details:</b>  <code>getWorkingDirectory</code> creates a new <code>Path</code> instance based on the VM's current working directory and returns it as an absolute path.</p>

	@return the current directory
	**********************************************************************/

	public static Path getWorkingDirectory ()
	{
		return createFromJavaFile (new File (getWorkingDirectoryAsString ()));
	}



	/**********************************************************************
	Returns working directory as string.

	<p><b>Details:</b>  <code>getWorkingDirectoryAsString</code> returns a string representing the VM's current working directory.</p>

	@return the working directory
	**********************************************************************/

	private static String getWorkingDirectoryAsString ()
	{
		return System.getProperty ("user.dir");
	}



	/**********************************************************************
	Converts File to Path.

	<p><b>Details:</b>  <code>createFromJavaFile</code> converts instances of <code>java.io.File</code> into <code>Path</code>s, recognizing the file system requirements of the host VM's platform.  If the current platform is not Unix, the path will be unixized as described in the class notes.</p>

	@param file the File
	@return the Path
	**********************************************************************/

	public static final Path createFromJavaFile (File file)
	{
		if (WINDOWS ())
			return fromWindows (file);
		return fromUnix (file);
	}



	/**********************************************************************
	Converts Unix File to Path.

	<p><b>Details:</b>  <code>fromUnix</code> fulfills the contract of <code>createFromJavaFile</code>, but does so assuming that the current platform is Unix.</p>

	@param unixfile the Unix File to convert
	@return the converted Path
	**********************************************************************/

	private static final Path fromUnix (File unixfile)
	{
//System.out.println("fromUnix");
		return new Path (unixfile . getPath ());
	}



	/**********************************************************************
	Converts Windows File to Path.

	<p><b>Details:</b>  <code>fromWindows</code> completes the contract of <code>createFromJavaFile</code>, but does so under the assumption that the current platform is Windows.</p>

	@param winfile the Windows File
	@return the Path
	**********************************************************************/

	private static final Path fromWindows (File winfile)
	{
//System.out.println("fromWindows");
		if (! WINDOWS ())
			throw new lava.UnreachableCodeException ();
		String winpath = winfile . getPath ();
		final int length = winpath . length ();
		if (length == 0)
			throw new IllegalArgumentException ("winpath=" + winpath);
		winpath = winpath . replace ('\\', '/');
		if (winpath . length () >= 2 &&	winpath . charAt (1) == ':')
		{
			char driveletter = winpath . charAt (0);
			if ('a' <= driveletter && driveletter <= 'z')
				driveletter = (char) (driveletter - 'a' + 'A');
			else if (driveletter < 'A' || 'Z' < driveletter)
				throw new IllegalArgumentException ("winpath=" + winpath);
			if (winpath . length () >= 3 && winpath . charAt (2) == '/')
			{
				// C:/file
				winpath = "/" + driveletter + "/" + winpath . substring (3);
			}
			else
			{
				// C:file
				return fromWindows (winfile . getAbsoluteFile ());
			}
		}
		else
		{
			if (winpath . charAt (0) == '/')
			{
				if (length > 1 && winpath . charAt (1) == '/')
				{
					winpath = "/NET/" + winpath . substring (2);
				}
// Uncomment these lines to disallow mounted working drive extension.
//				else
//				{
//					// /file
//					return fromWindows (winfile . getAbsoluteFile ());
//				}
			}
			else
			{
				// file
			}
		}
		return new Path (winpath);
	}



	//////////////////
	//              //
	//  toJavaFile  //
	//              //
	//////////////////



	public File toJavaFile ()
	{
		if (WINDOWS ())
			return toWindowsFile ();
		return toUnixFile ();
	}



	File toUnixFile ()
	{
		return new File (pathname);
	}



	private File toWindowsFile ()
	{
		return toWindowsFile (pathname);
	}



	private static File toWindowsFile (String pathname)
	{
		pathname = pathname . replace ('/', '\\');
		// If it's relative then just replace slashes with backslashes:
		if (pathname . charAt (0) != '\\')
			return new File (pathname);
		// OK, it's not relative.  Now what?
		int length = pathname . length ();
		switch (length)
		{
		case 0:
			// The constructors should never allow an empty path string.
			throw new lava.UnreachableCodeException ();
		case 1:
			// pathname == /
			return new File ("\\");
		case 2:
			// pathname == /x
			{
				char driveletter = pathname . charAt (1);
				if ('A' <= driveletter && driveletter <= 'Z')
					return new File (driveletter + ":\\");
				return new File (pathname);
			}
		default:
			if (pathname . equals ("\\NET"))
				return new File ("\\\\");
			if (pathname . startsWith ("\\NET\\"))
				return new File ("\\" + pathname . substring (4));
			if (pathname . charAt (2) == '\\')
			{
				// pathname == /x/...
				char driveletter = pathname . charAt (1);
				if ('A' <= driveletter && driveletter <= 'Z')
					return new File (driveletter + ":" + pathname . substring (2));
			}
			return new File (pathname);
		}
	}



	/////////////////////
	//                 //
	//  miscellaneous  //
	//                 //
	/////////////////////



	/**********************************************************************
	Root directory.

	<p><b>Details:</b>  <var>root_dir</var> is a predefined Path representing the root directory.  It is returned by <code>getRootDirectory</code>.</p>
	**********************************************************************/

	private static Path root_dir = new Path ("/");



	/**********************************************************************
	Returns root directory.

	<p><b>Details:</b>  getRootDirectory returns a Path representing the root directory.</p>

	@return the root directory
	**********************************************************************/

	public static Path getRootDirectory ()
	{
		return root_dir;
	}



	private static boolean WINDOWS ()
	{
		return Platform.getOsGenre () == Platform.WINDOWS;
	}



	private boolean isBrowsingPath ()
	{
		// assert: run in Windows only
		if (pathname . equals ("/NET"))
			return true;
		if (pathname . startsWith ("/NET/") && StringToolbox.count (pathname, '/') == 2)
			return true;
		return false;
	}



	////////////////////////////////////////////////
	//                                            //
	//  queries not requiring file system access  //
	//                                            //
	////////////////////////////////////////////////



	/**********************************************************************
	Tells whether path is absolute.

	<p><b>Details:</b>  <code>isAbsolute</code> returns <code>true</code> if this path is absolute, <code>false</code> otherwise.  A path is absolute if and only if its name begins with a slash ('<code>/</code>').</p>

	@return true iff path is absolute
	**********************************************************************/

	public final boolean isAbsolute ()
	{
		return PathToolbox.isAbsolutePath (pathname);
	}



	public Path getParentDirectory ()
	{
		return new Path (pathname + "/..");
	}



	public String getFileName ()
	{
		return PathToolbox.getFilename (pathname);
	}



	/**********************************************************************
	Converts to absolute path.

	<p><b>Details:</b>  <code>toAbsolutePath</code> creates a new, absolute path based on this path and returns it.  This path is resolved using the current working directory.</p>

	@return the absolute path
	**********************************************************************/

	public Path toAbsolutePath ()
	{
		return new Path (getWorkingDirectory () . pathname, pathname, CONDITIONAL_RELATIVE);
	}



	/**********************************************************************
	Converts to relative path.

	<p><b>Details:</b>  If this <code>Path</code> is relative, <code>toRelativePath</code> simply returns <code>this</code>.  Otherwise, <code>toRelativePath</code> determines the shortest relative <code>Path</code> which, when resolved from <var>wd</var>, indicates the same <code>Path</code> as this <code>Path</code></p>

	@param wd the "working" directory
	@return the relative Path
	**********************************************************************/

	public Path toRelativePath (Path wd)
	{
		if (isRelative ())
			return this;
		return new Path (PathToolbox.toRelativePath (wd . pathname, pathname));
	}



	/**********************************************************************
	Converts to relative path.

	<p><b>Details:</b>  This method is the same as toRelativePath(Path), except that the VM's current working directory is supplied as the first parameter.</p>
	**********************************************************************/

	public Path toRelativePath ()
	{
		return toRelativePath (getWorkingDirectory ());
	}



	public boolean isRelative ()
	{
		return ! isAbsolute ();
	}



	/////////////////////////
	//                     //
	//  query file system  //
	//                     //
	/////////////////////////



	public Path getCanonicalPath () throws IOException
	{
		EXISTS ();
		File jfile = toJavaFile ();
		jfile = jfile . getCanonicalFile ();
		return createFromJavaFile (jfile);
	}



	/**********************************************************************
	Tests existance.

	<p><b>Details:</b>  <code>exists</code> returns <code>true</code> if the file targeted by this path exists, <code>false</code> otherwise.  If this path is relative, it is first resolved using the current working directory.</p>

	<p>If exists cannot convert this path into system-<wbr>dependent File object, this method automatically returns false.  Note that this means in Windows, paths that don't begin with "/X", where X is a drive letter, never exist.</p>

	<p><b>Windows notes:</b> Note that in Java, there is no way (short of native methods) to detect whether a given computer exists on the network.  Thus, paths of the form "<tt>/NET/SERVERNAME</tt>" are considered by <code>exists</code> to always exist.  Similarly, there is no way to determine whether network browsing is enabled.  Therefore, <code>exists</code> simply returns a default value of <code>true</code> if this path is "<tt>/NET</tt>".  Of course, <code>exists</code> also returns <code>true</code> for "<tt>/</tt>".</p>

	@return true if the file exists
	**********************************************************************/

	public boolean exists () throws IOException
	{
		if (WINDOWS ())
			return existsInWindows ();
		File jfile = toJavaFile ();
		if (jfile == null)
			return false;
		return jfile . exists ();
	}



	private boolean existsInWindows () throws IOException
	{
		// /
		if (pathname . equals ("/"))
			return true;
		// /NET
		// /NET/SERVERNAME
		if (isBrowsingPath ())
			return true;
		File jfile = toWindowsFile ();
		if (jfile == null)
			return false;
		if (! jfile . exists ())
			return false;
		jfile = new Path
		(
			PathToolbox.simplifyPath
			(
				PathToolbox.resolveRelativePath
				(
					getWorkingDirectory () . pathname,
					pathname
				)
			)
		) . toWindowsFile ();
		// Check name case:
		return jfile . getAbsolutePath () . equals (jfile . getCanonicalPath ());
	}



	private void EXISTS () throws IOException
	{
		if (exists ())
			return;
		throw new FileNotFoundException (pathname);
	}



	/**********************************************************************
	Determines whether path is directory.

	<p><b>Details:</b>  <code>isDirectory</code> determines whether the file targeted by this path can be <em>treated</em> as a directory.  (In Unix-<wbr>speak, this means that soft links to directories are OK.)  If it can, then <code>isDirectory</code> returns <code>true</code>.  If it can not be, or if the file doesn't exist, <code>isDirectory</code> returns <code>false</code>.</p>

	<p>If a platform-<wbr>specific <code>File</code> representation of this <code>Path</code> cannot be created, this method automatically returns <code>false</code> (with one exception for Windows).  Otherwise, this method returns the result of calling <code>isDirectory</code> on the equivalent <code>File</code> object.</p>

	@return true if path targets an existing directory
	**********************************************************************/

	public boolean isDirectory () throws IOException
	{
		if (! exists ())
			return false;
		File jfile = toJavaFile ();
		if (jfile == null)
			return false;
		return jfile . isDirectory ();
	}



	/**********************************************************************
	Determines whether path is file.

	<p><b>Details:</b>  <code>isDataFile</code> determines whether the file targeted by this path can be <em>treated</em> as a normal data file.  If it can, then <code>isDataFile</code> returns <code>true</code>.  If it can not be, or if the file doesn't exist, then <code>isDataFile</code> returns <code>false</code>.</p>

	<p>If a platform-<wbr>specific <code>File</code> representation of this <code>Path</code> cannot be created, this method returns <code>false</code>.  Otherwise, this method returns the result of calling <code>isFile</code> on the equivalent <code>File</code> object.</p>

	@return true if path targets an existing data file
	**********************************************************************/

	public boolean isDataFile () throws IOException
	{
		if (! exists ())
			return false;
		File jfile = toJavaFile ();
		if (jfile == null)
			return false;
		return jfile . isFile ();
	}



	/**********************************************************************
	Determines whether file is readable.

	<p><b>Details:</b>  First, <code>isReadable</code> returns <code>false</code> if <code>exists</code> would return false.  Otherwise, <code>isReadable</code> determines whether the file is a <b>directory</b> or a <b>data file</b>.  If the file is a <b>directory</b>, <code>isReadable</code> returns <code>true</code> if and only if the directory's contents can be read.  If the file is a <b>data file</b>, <code>isReadable</code> returns <code>true</code> if and only if the file's contents can be read.</p>

	<p>This method is similar to <code>java.io.File.canRead</code>, except that this implementation actually works.  Note, in particular, that the Windows version of <code>File.canRead</code> does not always tell the truth.  This implementation, however, does not lie, since in Windows it performs an actual read test on the file.</p>

	<p>The Unix implementation also provides full functionality but does not require a test-<wbr>read, since permissions in the meta-<wbr>data provide the necessary clues to determine readability.</p>

	@return true if the file or directory is readable
	**********************************************************************/

	public boolean isReadable () throws IOException
	{
		if (! exists ())
			return false;
		if (WINDOWS ())
			return isReadableInWindows ();
		return isReadableInUnix ();
	}



	private boolean isReadableInUnix ()
	{
		File jfile = toUnixFile ();
		if (jfile == null)
			return false;
		return jfile . canRead ();
	}



	private boolean isReadableInWindows ()
	{
		File jfile = toWindowsFile ();
		if (jfile == null)
			return false;
		if (jfile . isDirectory ())
		{
			String[] list = jfile . list
			(
				new FilenameFilter ()
				{
					public boolean accept (File ignored1, String ignored2)
					{
						return false;
					}
				}
			);
			return list != null;
		}
		if (jfile . isFile ())
		{
			// shortcut
			if (! jfile . canRead ())
				return false;
			// real test
			try
			{
				FileInputStream in = new FileInputStream (jfile);
				IoCloser.close (in);
			}
			catch (IOException e)
			{
				return false;
			}
			return true;
		}
		return jfile . canRead ();
	}



	/**********************************************************************
	Determines whether file is writable.

	<p><b>Details:</b>  First, <code>isWritable</code> returns <code>false</code> if <code>exists</code> would return false.  Otherwise, <code>isWritable</code> determines whether the file is a <b>directory</b> or a <b>data file</b>.  If the file is a <b>directory</b>, <code>isWritable</code> returns <code>true</code> if and only if the directory can be deleted (assuming it is empty).  If the file is a <b>data file</b>, <code>isWritable</code> returns <code>true</code> if and only if the file's contents can be updated.</p>

	<p>This method is similar to <code>java.io.File.canWrite</code>, except that its specification is more clear.</p>

	<p><b>Unix notes:</b> In Unix, if a directory is not writable then not only can it not be deleted, but files inside it cannot be deleted or created either.  Already existing files in the directory can still be updated, however.</p>

	<p><b>Windows notes:</b> In Windows, directories marked as "read only" usually resist being deleted, but do not necessarily resist attempts to create new files or delete existing files within them.  There may be exceptions in the case of shared directories or mapped network drives, however.  In Windows, it may not be possible to determine whether a directory's file table is truly read only without actually attempting to create or delete a file.  If a directory shared by a Samba server appears to not be writable, as indicated by this method, then it is probably the case that the file table is not writable as well.</p>

	@return true if the file or directory is readable
	**********************************************************************/

	public boolean isWritable () throws IOException
	{
		if (! exists ())
			return false;
		File jfile = toJavaFile ();
		if (jfile == null)
			return false;
		return jfile . canWrite ();
	}



	/**********************************************************************
	Returns file size.

	<p><b>Details:</b>  <code>getSize</code> returns the size of this file, if it exists, or throws a <code>FileNotFoundException</code> if it does not.  If this is a <code>directory</code>, <code>getSize</code> returns -1.</p>

	<p>This method is similar to <code>java.io.File.length</code>, except that its behavior is consistent across platforms with respect to the sizes reported for directories.  In Unix, for example, <code>length</code> returns the size of the directory's inode table, but in Windows, <code>length</code> returns 0 regardless of the file table size.  This is inconsistent behavior that detracts from the portability of Java applications.</p>

	<p>This method takes a stand on what ought to be returned and returns it.</p>

	@return the size of the file
	@exception FileNotFoundException if the file does not exist
	**********************************************************************/

	public long getSize () throws IOException
	{
		EXISTS ();
		File javafile = toJavaFile ();
		if (javafile . isDirectory ())
			return -1;
		return javafile . length ();
	}



	/**********************************************************************
	Returns modification timestamp.

	<p><b>Details:</b>  <code>getLastModifiedTime</code> returns the time, in milliseconds since the 1970 epoch, that this file was last modified.  If this file is a directory and the platform is Unix, the last modified time indicates the time of last modification to the directory's inode structure.  In Windows, the last modified time usually refers to the time the directory was created.</p>

	<p><code>getLastModifiedTime</code> throws a <code>FileNotFoundException</code> if the file does not exist.</p>

	@return the modification timestamp
	@exception FileNotFoundException if the file does not exist
	**********************************************************************/

	public long getLastModifiedTime () throws IOException
	{
		EXISTS ();
		File javafile = toJavaFile ();
		return javafile . lastModified ();
	}



	/**********************************************************************
	Lists directory contents.

	<p><b>Details:</b>  <code>list</code> treats this path as a directory and attempts to list its contents.  If this path does not exist or is not a directory, or if it is a diretory but is not readable, <code>list</code> throws an <code>IOException</code>.  Otherwise, the contents of this diretory are returned in an array of <code>Path</code>s.  If this path is relative, it is first resolved using the current working directory.  In either case, each element in the returned array is automatically represented as an absolute path, since each element refers to an actually existing file.</p>

	<p><b>Windows notes:</b> If this path is "/", <code>list</code> returns a drive list, in the form of {"/A", "/C", "/D", ...}.  Included in this list is also the Network Neighborhood virtual directory, "/NET".  If this path is "/NET" or "/NET/servername", however, list returns an empty list, since browsing is not possible in pure Java code.  (Actually, it <em>is</em> possible, but Sharky is not willing to implement an SMB client in Java.  Sorry.  Any volunteers?)</p>

	@return the directory contents
	**********************************************************************/

	public Path[] list () throws IOException
	{
		EXISTS ();
		if (! isDirectory ())
			throw new DirectoryNotFoundException (pathname);
		Path[] list;
		if (WINDOWS ())
			list = listInWindows ();
		else
			list = listInJava ();
		Arrays.sort (list);
		return list;
	}



	private static final Object dummy_value = new Object ();



	private Path[] listInWindows () throws IOException
	{
		if (isBrowsingPath ())
			return new Path [0];
		Path[] paths = listInJava ();
		if (pathname . equals ("/"))
		{
			Hashtable ht = new Hashtable ();
			// Add native dir listing to set:
			int size = paths . length;
			for (int i = 0; i < size; ++ i)
				ht . put (paths [i], dummy_value);
			// Add drive list to set:
			File[] files = File.listRoots ();
			if (files == null)
				throw new IOException ("can't list FS roots");
			size = files . length;
			for (int i = 0; i < size; ++ i)
				ht . put (fromWindows (files [i]), dummy_value);
			// Convert set to array:
			size = ht . size ();
			paths = new Path [size];
			Enumeration htkeys = ht . keys ();
			for (int i = 0; i < size; ++ i)
				paths [i] = (Path) htkeys . nextElement ();
		}
		return paths;
	}



	private Path[] listInJava () throws IOException
	{
		File javafile = toJavaFile ();
//		String abspath = javafile . getAbsolutePath ();
		String[] slist = javafile . list ();
		if (slist == null)
			throw new IOException (pathname + " is not a readable directory");
		int length = slist . length;
		Path[] list = new Path [length];
		for (int i = 0; i < length; ++ i)
			list [i] = new Path (pathname + '/' + slist [i]);
//		Vector v = new Vector ();
//		for (int i = 0; i < length; ++ i)
//			v . addElement (createFromJavaFile (new File (abspath, slist [i])));
//		Path[] list = new Path [v . size ()];
//		v . copyInto (list);
		return list;
	}



	//////////////////////////
	//                      //
	//  update file system  //
	//                      //
	//////////////////////////



	/**********************************************************************
	Creates file.

	<p><b>Details:</b>  <code>createAsFile</code> attempts to create a zero-<wbr>length file whose location is identified by this path.  If the file cannot be created, <code>createAsFile</code> throws an <code>IOException</code>.  Possible reasons for failure include:</p>

	<ul>
		<li>The parent directory does not exist (see note below).
		<li>The parent directory, volume, or partition is read-<wbr>only.
		<li>A directory or file by the same name already exists.
	</ul>

	<p>If <var>mkdir</var> is <code>true</code>, any necessary parent directories will automatically be created.</p>

	<p><code>createAsFile</code> throws an <code>IOException</code> if the file cannot be created.</p>

	@param mkdir whether or not parent directories should be created
	@exception IOException if the directory cannot be created
	**********************************************************************/

	public void createAsFile (boolean mkdir) throws IOException
	{
		Path pd = getParentDirectory ();
		if (mkdir)
			pd . toJavaFile () . mkdirs ();
		if (! pd . exists ())
			throw new IOException ("unable to create parent directories");
		File file = toJavaFile ();
		if (! file . createNewFile ())
			throw new IOException ("unable to create file " + pathname);
	}



	/**********************************************************************
	Creates file.

	<p><b>Details:</b>  <code>createAsDirectory</code> attempts to create a directory whose location is identified by this path.  If the directory cannot be created, <code>createAsDirectory</code> throws an <code>IOException</code>.  Possible reasons for failure include:</p>

	<ul>
		<li>The parent directory does not exist (see note below).
		<li>The parent directory, volume, or partition is read-<wbr>only.
		<li>A directory or file by the same name already exists.
	</ul>

	<p>If <var>mkdir</var> is <code>true</code>, any necessary parent directories will automatically be created.</p>

	<p><code>createAsDirectory</code> throws an <code>IOException</code> if the directory cannot be created.</p>

	@param mkdir whether or not parent directories should be created
	@exception IOException if the file cannot be created
	**********************************************************************/

	public void createAsDirectory (boolean mkdir) throws IOException
	{
		File file = toJavaFile ();
		boolean passed;
		if (mkdir)
			passed = file . mkdirs ();
		else
			passed = file . mkdir ();
		if (! (passed && exists ()))
			throw new IOException ("unable to create directory " + pathname);
	}



	/**********************************************************************
	Deletes file or directory.

	<p><b>Details:</b>  <code>unlink</code> removes the directory or file represented by this path.  If this path is a directory, then the directory must be empty for the unlink operation to succeed.  If <code>unlink</code> fails, an IOException is thrown.</p>

	@exception IOException if the file or directory cannot be deleted
	**********************************************************************/

	public void unlink () throws IOException
	{
		EXISTS ();
		if (! toJavaFile () . delete ())
			throw new IOException ("can't unlink file " + pathname);
	}



	public void setLastModifiedTime (long new_time) throws IOException
	{
		EXISTS ();
		boolean passed = toJavaFile () . setLastModified (new_time);
		if (! passed)
			throw new IOException ("unable to set modified time for file " + pathname);
	}



	/**********************************************************************
	Moves file.

	<p><b>Details:</b>  move moves this file or directory to the target file or directory.  If the target is an existing directory, this file will be placed in that directory under its same name.  Otherwise, move will attempt to move and/or rename this file or directory so that it has the same path as dest.  In short, this method behaves like the Unix mv command.</p>

	<p>If the target (dest) exists and is not a directory, the file already at dest will be overwritten by this file.  If the operation fails, an IOException will be thrown.</p>

	<!-- What happens if the target is a file but the source is a directory? -->

	@param dest the target
	@exception IOException if the operation cannot complete
	**********************************************************************/

	public void move (Path dest) throws IOException
	{
		EXISTS ();
		if (WINDOWS ())
			moveInWindows (this, dest);
		else
			moveInUnix (dest);
	}



	private static void moveInWindows (Path src, Path dest) throws IOException
	{
		// Move file src to nai  dest: src moved to dest's parent and gets dest's name
		// Move dir  src to nai  dest: src moved to dest's parent and gets dest's name
		// Move file src to dir  dest: src moved to dest*
		// Move dir  src to dir  dest: src moved to dest*
		// Move file src to file dest: src replaces dest
		// Move dir  src to file dest: src replaces dest
		// *Fails if dir has a subdirectory named src.
		src  = src  . toAbsolutePath ();
		dest = dest . toAbsolutePath ();
		File srcfile  = src  . toJavaFile ();
		File destfile = dest . toJavaFile ();
		if (! dest . exists ())
		{
			if (destfile . exists ())
				throw new IOException ("case conflict in dest");
			Path destpar = dest . getParentDirectory ();
			if (! destpar . exists ())
				throw new IOException ("parent directory does not exist");
		}
		else if (destfile . isDirectory ())
		{
			dest = new Path (dest, src . getFileName (), FORCE_RELATIVE);
			destfile = dest . toJavaFile ();
			if (! dest . exists ())
			{
				if (destfile . exists ())
					throw new IOException ("case conflict in dest");
			}
			else
			{
				if (destfile . isDirectory ())
					throw new IOException ("can't replace directory");
				else
					if (! destfile . delete ())
						throw new IOException ("unable to replace dest");
			}
		}
		else
		{
			if (! destfile . delete ())
				throw new IOException ("unable to replace dest");
		}
//System.out.println("srcfile: " + srcfile);
//System.out.println("destfile: " + destfile);
		if (! srcfile . renameTo (destfile))
			throw new IOException ("move failed");
	}



	private void moveInUnix (Path dest) throws IOException
	{
		if (! toUnixFile () . renameTo (dest . toUnixFile ()))
			throw new IOException ("move failed");
//		if (dest . isDirectory ())
//			dest = new Path (dest . pathname, getFileName (), FORCE_RELATIVE);
//		// TODO: check for moveing file into directory
//		if (! toJavaFile () . renameTo (dest . toJavaFile ()))
//			throw new IOException ("unable to move " + pathname);
	}



	//////////////////////
	//                  //
	//  Object methods  //
	//                  //
	//////////////////////



	/**********************************************************************
	Tests for equivalence.

	<p><b>Details:</b>  equals fulfills its contract as described in java.lang.Object.  Here are the relevant implementation details:</p>

	<ol>
		<li>If o is null or is not a Path, return false.
		<li>If o is relative but this is absolute, or vice-<wbr>versa, return false.
		<li>Otherwise, return true if the paths match.
	</ol>

	@param o the object to compare
	@return true iff o represents the same path as this
	**********************************************************************/

	public boolean equals (Object o)
	{
		if (! (o instanceof Path))
			return false;
		return pathname . equals (((Path) o) . pathname);
	}



	/**********************************************************************
	Returns hash code.

	<p><b>Details:</b>  hashCode fulfills its contract as described in java.lang.Object.</p>
	**********************************************************************/

	public int hashCode ()
	{
		return ~ pathname . hashCode ();
	}



	/**********************************************************************
	Returns path.

	<p><b>Details:</b>  <code>toString</code> returns a string representation of this Path.  This may not be the same value provided at construction time, but it will be equivalent.</p>

	@return the path
	**********************************************************************/

	public String toString ()
	{
		return pathname;
	}



	public int compareTo (Object o2)
	{
		return pathname . compareTo (((Path) o2) . pathname);
	}



	/**********************************************************************
	Corrects case to match existing files.

	<p><b>Details:</b>  On non-Windows platforms, adjustCase simply returns <code>this</code>.  On Windows, adjust case corrects the case of characters in the filename to match the case of actual directory and file names found on the hard drive.  Because Path is case-sensitive, adjustCase must be used to convert nearly-matching Paths into matching paths if case-sensitive behavior is not desired.</p>

	@return adjusted Path

	@since 2001.03.09
	**********************************************************************/

	public Path adjustCase ()
	{
		if (! WINDOWS ())
			return this;
		try
		{
			String file = toJavaFile () . getCanonicalPath () . replace ('\\', '/');
			String common = StringToolbox.getCommonSuffix (file . toUpperCase (), pathname . toUpperCase ());
			return new Path
			(	pathname . substring
				(	0,
					pathname . length () - common . length ()
				)
			+	StringToolbox.right
				(	file,
					common . length ()
				)
			);
		}
		catch (IOException e)
		{
			return this;
		}
	}



}



