///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.File;
import java.io.IOException;
import java.util.Stack;
import java.util.Vector;
import java.util.Enumeration;
import lava.io.Path;
import java.util.Hashtable;



/******************************************************************************
Unique file list manager.

<p><b>Details:</b> PathSet is an ordered file set manager designed to help you generate a file list from user-supplied arguments.  It allows you to add and subtract sets of files either by naming the individual files or by naming directories that can be searched recursively.</p>

<p><code>PathSet</code> detects duplicate entries and automatically keeps them out of the set.  <code>PathSet</code> even detects when two apparently different paths represent the same path due to "symlink magic."  <code>PathSet</code> also preserves the order in which the files were added.  When two equivalent paths are added, the first path added gets priority, unless the first is removed before the second is added.</p>

<p>After a set is formed, iterator methods can be used to walk through the ordered list of files.</p>

@since 2000.07.08
@version 2001.03.09
@author Sharky
******************************************************************************/

public class PathSet
{



	private Hashtable seen = new Hashtable ();



	private Vector set = new Vector ();



	private boolean dirs_allowed;



	/**********************************************************************
	Initializes empty set.

	<p><b>Details:</b>  This constructor initializes an empty set and determines whether directory paths (as opposed to actual files) can be added to the set.  If <code>dirs_allowed</code> is true, the set will accept directory paths.  Otherwise, directory paths will be automatically discarded when added.</p>

	@param dirs_allowed
	**********************************************************************/

	public PathSet (boolean dirs_allowed)
	{
		this . dirs_allowed = dirs_allowed;
	}



	/**********************************************************************
	Appends path to set.

	<p><b>Details:</b>  <code>add</code> appends the given path (<var>path</var>) to the end of the set if the path is not already a member of the set.  <var>path</var> will not be retained in the set if it is a directory and <var>dirs_allowed</var> is <code>false</code>.</p>

	<p><b>Polymorphism warning:</b> This method is called by <code>add(String)</code> and <code>addRecursively(Path)</code>.</p>

	@param path the path to add
	@exception IOException if file examination fails
	**********************************************************************/

	public void add (Path path) throws IOException
	{
		Path key = path . getCanonicalPath ();
		if (seen . containsKey (key))
			return;
		if (path . isDirectory () && ! dirs_allowed)
			return;
		int index = set . size ();
		seen . put (key, new Integer (index));
		set . add (path);
	}



	/**********************************************************************
	Convenience method.

	<p><b>Details:</b>  This convenience method converts the given <code>File</code> to a <code>Path</code> and calls <code>add(Path)</code>.</p>
	**********************************************************************/

	public final void add (File file) throws IOException
	{
		add (Path.createFromJavaFile(file) . adjustCase ());
	}



	/**********************************************************************
	Appends directory to set.

	<p><b>Details:</b>  <code>addRecursively</code> appends the given <code>Path</code> and, if it is a directory, all of the <code>Path</code>s underneath it, recursively, to the set.  Some paths may not be added if they represent directories and <var>allow_dirs</var> is false.</p>

	<p><b>Polymorphism warnings:</b> This method calls add(Path) for each individual path added to the set.  This method is called by <code>addRecursively(String)</code>.</p>

	@param path the path to add
	@exception IOException if file examination fails
	**********************************************************************/

	public void addRecursively (Path path) throws IOException
	{
		Stack stack = new Stack ();
		stack . push (path);
		while (! stack . isEmpty ())
		{
			path = (Path) stack . pop ();
			add (path);
			if (path . isDirectory ())
			{
				Path[] list = path . list ();
				for (int i = list . length - 1; i >= 0; -- i)
					stack . push (list [i]);
			}
		}
	}



	/**********************************************************************
	Convenience method.

	<p><b>Details:</b>  This convenience method converts the given <code>File</code> to a <code>Path</code> and calls <code>addRecursively(Path)</code>.</p>
	**********************************************************************/

	public final void addRecursively (File file) throws IOException
	{
		addRecursively (Path.createFromJavaFile(file) . adjustCase ());
	}



	/**********************************************************************
	Removes path from set.

	<p><b>Details:</b>  <code>remove</code> removes the given Path (<var>path</var>) from this set if it is currently in the set.  If <var>path</var> is not in the set, this method has no effect.</p>

	<p><b>Polymorphism warning:</b> This method is called by <code>remove(String)</code> and <code>removeRecursively(Path)</code></p>

	@param path the path to remove
	**********************************************************************/

	public void remove (Path path) throws IOException
	{
		Path key = path . getCanonicalPath ();
		Integer index = (Integer) seen . get (key);
		if (index == null)
			return;
		set . setElementAt (null, index . intValue ());
		seen . remove (key);
	}



	/**********************************************************************
	Convenience method.

	<p><b>Details:</b>  This convenience method converts the given <code>File</code> to a <code>Path</code> and calls <code>remove(Path)</code>.</p>
	**********************************************************************/

	public void remove (File file) throws IOException
	{
		remove (Path.createFromJavaFile(file) . adjustCase ());
	}



	/**********************************************************************
	Removes directory from set.

	<p><b>Details:</b>  <code>remove</code> removes the given <code>Path</code> (<var>path</var>) and, if it is a directory, all of the <code>Path</code>s underneath it, recursively, from this set.</p>

	<p><b>Polymorphism warnings:</b> This method calls <code>remove(Path)</code> for each individual <code>Path</code> to be removed from the set.  This method is called by <code>removeRecursively(String)</code>.</p>

	@param path the path to remove
	**********************************************************************/

	public void removeRecursively (Path path) throws IOException
	{
		Stack stack = new Stack ();
		stack . push (path);
		while (! stack . isEmpty ())
		{
			path = (Path) stack . pop ();
			remove (path);
			if (path . isDirectory ())
			{
				Path[] list = path . list ();
				for (int i = list . length - 1; i >= 0; -- i)
					stack . push (list [i]);
			}
		}
	}



	/**********************************************************************
	Convenience method.

	<p><b>Details:</b>  This convenience method converts the given <code>File</code> to a <code>Path</code> and calls <code>removeRecursively(Path)</code>.</p>
	**********************************************************************/

	public void removeRecursively (File file) throws IOException
	{
		removeRecursively (Path.createFromJavaFile(file) . adjustCase ());
	}



	private void compress ()
	{
		int size = set . size ();
		int i = 0;
		for (int j = 0; j < size; ++ j)
		{
			Object o = set . elementAt (j);
			if (o != null)
				set . setElementAt (o, i ++);
		}
		set . setSize (i);
	}



	/**********************************************************************
	Enumerates set.

	<p><b>Details:</b>  enumerate returns and Enumeration that lists every Path currently in this set.</p>

	@return the Enumeration

	@since 2001.03.09
	**********************************************************************/

	public Enumeration enumerate ()
	{
		synchronized (set)
		{
			compress ();
			return ((Vector) set . clone ()) . elements ();
		}
	}



}



