///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import lava.Platform;
import lava.string.StringToolbox;



/******************************************************************************
ICMP-based ping utilities.

<p><b>Details:</b> <code>Ping</code> is a collection of functions for performing low level, ICMP-based ping tests on remote hosts.</p>

<p>Presently, <code>Ping</code> contains only one function, countReplies.  More will be added later.  If you have requests, please submit them to Sharkysoft.</p>

<p>Warning: <code>Ping</code> uses a few platform-dependent tricks.  You should verify that this class works on your target platform before integrating it into your application.  Run the stand-alone test by executing this class from the command line:</p>

<blockquote>
	<code>java lava.io.Ping &lt;host&gt; [&lt;count&gt;]</code>
</blockquote>

<p><code>Ping</code> has been tested and found operational on the following platforms:</p>

<ul>
	<li>Windows 2000</li>
	<li>Red Hat Linux 7.0</li>
</ul>

<p>If you test <code>Ping</code> and find that it works on another platform, please let Sharkysoft know so we can add the platform to the above list.</p>

@since 2000.10.21
@author Sharky
******************************************************************************/

public class Ping
{



	/**********************************************************************
	Counts replies from pinged host.

	<p><b>Details:</b> countReplies pings the given host (<var>host</var>) <var>tries</var> times.  The replies are counted and returned.</p>

	@param host host to ping
	@param tries number of pings to send
	@return number of ping replies received
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public static int countReplies (String host, int tries) throws IOException
	{
		if (tries < 0)
			throw new IllegalArgumentException ("tries=" + tries);
		switch (Platform.getOsGenre ())
		{
		case Platform.WINDOWS:
			return countRepliesWindows (host, tries);
		case Platform.UNIX:
			return countRepliesLinux (host, tries);
		default:
			throw new lava.NotSupportedException ();
		}
	}



	private static int countRepliesWindows (String host, int tries) throws IOException
	{
		return countReplies
		(
			new String[]
			{
				"ping",
				"-n",
				String.valueOf (tries),
				host
			},
			"Reply from"
		);
	}



	private static int countRepliesLinux (String host, int tries) throws IOException
	{
		return countReplies
		(
			new String[]
			{
				"ping",
				"-c",
				String.valueOf (tries),
				host
			},
			"bytes from"
		);
	}



	private static int countReplies (String[] cmd, String hitstring) throws IOException
	{
		Process p = Runtime.getRuntime () . exec (cmd);
		BufferedReader br = new BufferedReader
		(
			new InputStreamReader
			(
				p . getInputStream ()
			)
		);
		int replies = 0;
		while (true)
		{
			String line = br . readLine ();
			if (line == null)
				break;
			if (StringToolbox.contains (line, hitstring))
				++ replies;
		}
		try
		{
			int exitcode = p . waitFor ();
		}
		catch (InterruptedException ignored)
		{
		}
//		System.out . println (replies + " replies.");
		return replies;
	}



	/**********************************************************************
	Test program entry point.

	<p><b>Details:</b>  main is the program entry point for the stand-alone compatibility test.</p>

	@param args command line arguments
	**********************************************************************/

	public static void main (String[] args) throws Exception
	{
		if (args . length == 0 || args . length > 2)
		{
			IoToolbox.printLines
			(
				new String[]
				{
					"lava.io.Ping, by Charlton Rose",
					"Copyright (c) 2000 Sharkysoft.  All rights reserved.",
					"usage: lava.io.Ping <host> [<count>]"
				}
			);
			return;
		}
		String host;
		int count = 4;
		switch (args . length)
		{
		case 2:
			count = Integer.parseInt (args [1]);
		case 1:
			host = args [0];
			break;
		default:
			throw new lava.UnreachableCodeException ();
		}
		System.out . println ("Pinging " + args [0] + " " + count + " times...");
		System.out . println (countReplies (host, count) + " replies received.");
	}



}



