///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import lava.io.StreamFormatException;
import java.io.PushbackReader;
import java.io.Reader;
import java.io.IOException;
import lava.string.StringToolbox;
import lava.clib.Ctype;
import lava.string.ICharacterClass;
import java.io.StringReader;



/******************************************************************************
Stream parsing utilities.

<p><b>Details:</b>  <code>StreamParser</code> contains a variety of methods for parsing elements from <code>PushbackReader</code>s.  Each method is designed to read and parse a certain type of object.  If the object sought by a particular parsing method is next in the stream, that method parses the object and returns it.  If the requested object is <em>not</em> immediately next in the stream, however, the parsing method returns a failure indicator (usually <code>null</code>).</p>

<p>In cases of success, the stream is left so that the next character to be read is the first character immediately after the object that was parsed.  In cases of failure, however, the stream is left as if nothing had been read.  This is possible only because the supplied stream supports <code>unread</code> and each method is configured to consume no more data than is necessary to define the parsed result.</p>

<p>Because <code>StreamParser</code>'s parsing methods use <code>unread</code> liberally, the supplied <code>PushbackReader</code>'s pushback buffer should be allocated large enough to handle the pushback resulting from failed parse attempts.  For example, suppose there was a method called "<code>parseFoo</code>" whose purpose was to parse grammatically correct sentences.  In some cases, <code>parseFoo</code> would need to read from the beginning of the sentence all the way to the sentence's terminating punctuation mark before it could determine that the sentence was malformed.  But then, <code>parseFoo</code> would be obligated to push all of the examined characters back onto the stream, and there might be a lot of them.  When you create PushbackReaders for use with this class, you should carefully consider the amount of pushback that may be required.</p>

@since 2000.01.29
@version 2000.07.07
******************************************************************************/

public class StreamParser
{



	///////////////////////
	//                   //
	//  quoted C string  //
	//                   //
	///////////////////////



	/**********************************************************************
	Expects quoted C string.

	<p><b>Details:</b>  expectQuotedCString does the same thing as tryQuotedCString but throws a StreamFormatException if the parsing is unsuccessful.</p>

	@param in the input source
	@return the C string
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static final String expectQuotedCString (PushbackReader in) throws IOException
	{
		String s = tryQuotedCString (in);
		if (s == null)
			throw new StreamFormatException ("expected quoted C string");
		return s;
	}



	/**********************************************************************
	Parses quoted C string.

	<p><b>Details:</b>  tryQuotedCString parses a C-<wbr>style string literal surrounded by either single quotes ('\'') or double quotes ('"').  If parsing succeeds, the entire quoted C string is returned, complete with surrounding quotes and undecoded, escaped characters.  If parsing fails, <code>null</code> is returned.</p>

	@param in the input source
	@return the parsed C string
	@exception IOException if an I/O error occurs

	@version 2000.01.29
	@since unknown
	**********************************************************************/

	public static final String tryQuotedCString (PushbackReader in) throws IOException
	{
		return tryQuotedCString (in, true);
	}



	/**********************************************************************
	Parses quoted C string.

	<p><b>Details:</b>  This method does the work of tryQuotedCString(PushbackReader), while allowing an additional argument for the C string's "well formedness."  The <var>wellformed</var> parameter is set to <code>true</code> if this parser must reject strings that span lines without properly escaping the newline character ('\n').  Otherwise, unescaped newline characters will be allowed.  (Only '<code>\n</code>' is recognized as a line separator.  Other characters, such as '\r', will be included in the string regardless of the value of <var>wellformed</var>.  If you want parseQuoted to reject other unescaped line separators, consider filtering the stream through <code>UnixLineReader</code>.)</p>

	@param in the input source
	@param wellformed whether string can include unescaped newline characters
	@return the parsed C string
	@exception IOException if an I/O error occurs

	@version 2000.01.29
	@since unknown
	**********************************************************************/

	private static final String tryQuotedCString (PushbackReader in, boolean wellformed) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
		int c;
		c = ub . read ();
		// If we're not starting with a single or double quote, forget it:
		if (c != '"' && c != '\'')
		{
			ub . unread ();
			return null;
		}
		// Hang onto the quote style so we can recognize string termination later:
		final char quote = (char) c;
		// Start reading until we're done!  As we read, everything that's read will accumulate in the UnreadBuffer:
		boolean escaping = false;
	read_loop:
		while (true)
		{
			// Get the char.  If it's EOF, abort:
			c = ub . read ();
			if (c < 0)
				break read_loop;
			switch (c)
			{
			// Allow newlines in wellformed strings only if escaped:
			case '\n':
				if (! escaping && wellformed)
					break read_loop;
				break;
			// Quote mark must mean end of string or it must be escaped:
			case '"':
			case '\'':
				if (escaping || c != quote)
					break;
				// We're done!
				return ub . getContents ();
			case '\\':
				// Process escape character:
				if (escaping)
					break;
				escaping = true;
				continue;
			}
			escaping = false;
		}
		ub . unreadAll ();
		return null;
	}



	//////////////
	//          //
	//  string  //
	//          //
	//////////////



	/**********************************************************************
	Expects a string.

	<p><b>Details:</b>  expectString does the same thing as tryString, but throws a StreamFormatException if the parsing operation fails.</p>

	@param in the input source
	@return the string
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static final String expectString (PushbackReader in) throws IOException
	{
		String s = tryString (in);
		if (s == null)
			throw new StreamFormatException ("expected string");
		return s;
	}



	/**********************************************************************
	Parses string.

	<p><b>Details:</b>  <code>tryString</code> parses and returns a string, where a string is defined as any continuous sequence of non-<wbr>space characters.  A character <var>c</var> is considered non-<wbr>space if and only if <code>Character.isWhitespace (c)</code> returns <code>false</code>.  If a string by this definition is not immediately next in the stream, <code>tryString</code> returns <code>null</code>.  Otherwise, <code>tryString</code> returns the string.</p>

	@param in the input source
	@return the string
	@exception IOException if an I/O error occurs

	@version 2000.01.29
	@since 1998.09.26
	**********************************************************************/

	public static final String tryString (PushbackReader in) throws IOException
	{
		StringBuffer buff = new StringBuffer ();
		boolean read_something = false;
		while (true)
		{
			int c = in . read ();
			if (c < 0)
				break;
			if (Character.isWhitespace ((char) c))
			{
				in . unread (c);
				break;
			}
			buff . append ((char) c);
			read_something = true;
		}
		if (! read_something)
			return null;
		return buff . toString ();
	}



	//////////////////////
	//                  //
	//  shell argument  //
	//                  //
	//////////////////////



	/**********************************************************************
	Expects shell argument.

	<p><b>Details:</b>  <code>expectShellArgument</code> does the same thing as <code>tryShellArgument</code> but throws a StreamFormatException if the parsing operation fails.</p>

	@param in the input source
	@return the parsed argument
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static String expectShellArgument (PushbackReader in) throws IOException
	{
		String arg = tryShellArgument (in);
		if (arg == null)
			throw new StreamFormatException ("expected shell argument");
		return arg;
	}



	/**********************************************************************
	Parses shell argument.

	<p><b>Details:</b>  <code>tryShellArgument</code> parses a single shell-<wbr>like token from the stream, such as a sequence of characters that might be considered one argument in a Unix command interpreter like <cite>bash</cite>.  <code>tryShellArgument</code> recognizes shell meta-<wbr>characters, single and double quotes, backslash escapes, etc.  <code>man bash</code> for more information.</p>

	@param in the input source
	@return the parsed argument
	@exception IOException if an I/O error occurs

	@since 2000.01.29
	@version 2000.06.01
	**********************************************************************/

	public static final String tryShellArgument (PushbackReader in) throws IOException
	{
		UnreadBuffer urb = new UnreadBuffer (in);
		StringBuffer parsed = new StringBuffer ();
		boolean got_something = false;
		while (true)
		{
			int c = urb . peek ();
			if (c < 0)
				break;
			String s;
			if (c == '"')
			{
				s = tryDoubleQuotedSegment (urb);
				if (s == null)
				{
					urb . unreadAll ();
					return null;
				}
				s = s . substring (1, s . length () - 1);
			}
			else if (c == '\'')
			{
				s = trySingleQuotedSegment (urb);
				if (s == null)
				{
					urb . unreadAll ();
					return null;
				}
			}
			else
			{
				s = tryUnquotedSegment (urb);
				if (s == null)
					break;
			}
			got_something = true;
			parsed . append (s);
		}
		if (got_something)
			return parsed . toString ();
		return null;
	}



	/**********************************************************************
	Parses unquoted portion of bash argument.

	<p><b>Details:</b>  <code>tryUnquotedSegment</code> is part of the <code>tryShellArgument</code> implementation.  This method parses an unquoted portion of an argument, returning it if successful, but also placing the characters it reads in urb.  This method asserts that an unquoted segment is next in the stream and thus should not be called unless that fact has already been verified by the caller.</p>

	@param in the input source
	@param urb the cummulative unread buffer
	@return the parsed result (decoded)
	@exception IOException if an I/O error occurs

	@since 2000.01.29
	**********************************************************************/

	private static String tryUnquotedSegment (UnreadBuffer urb) throws IOException
	{
		StringBuffer parsed = new StringBuffer ();
		boolean escaping = false;
		boolean read_something = false;
	parsing:
		while (true)
		{
			int c = urb . read ();
			if (c < 0)
			{
				if (escaping)
					return null;
				break parsing;
			}
			if (escaping)
			{
				if (c != '\n')
					parsed . append ((char) c);
				escaping = false;
				continue;
			}
			if (c == '\\')
			{
				escaping = true;
				continue;
			}
			switch (c)
			{
			default:
				if (! Character.isWhitespace ((char) c))
					break;
				// fall through
			case '"':
			case '\'':
			case '>':
			case '<':
			case '&':
			case '|':
			case '(':
			case ')':
			case ';':
				urb . unread ();
				break parsing;
			}
			parsed . append ((char) c);
		}
		if (parsed . length () > 0)
			return parsed . toString ();
		return null;
	}



	/**********************************************************************
	Parses single-quoted portion of bash argument.

	<p><b>Details:</b>  <code>trySingleQuotedSegment</code> parses a single-<wbr>quoted segment of a bash argument, beginning with the opening tick and ending after the closing tick.  If successful, this method returns the parsed result, null otherwise.  In either case, the characters consumed are pushed onto the supplied unread buffer.</p>

	@param in the input source
	@param urb the cumulative unread buffer
	@return the parsed result (decoded)
	@exception IOException if an I/O error occurs

	@since 2000.01.29
	**********************************************************************/

	private static String trySingleQuotedSegment (UnreadBuffer urb) throws IOException
	{
		StringBuffer parsed = new StringBuffer ();
		int c = urb . read ();
		if (c != '\'')
			return null;
	parsing:
		while (true)
		{
			c = urb . read ();
			if (c < 0)
				return null;
			if (c == '\'')
				break;
			parsed . append ((char) c);
		}
		return parsed . toString ();
	}



	/**********************************************************************
	Parses double-quoted portion of bash argument.

	<p><b>Details:</b>  <code>tryDoubleQuotedSegment</code> parses a double-<wbr>quoted segment of a bash argument, beginning with the opening quote and ending after the closing quote.  If successful, this method returns the parsed result, decoded, or null otherwise.  In either case, the characters consumed are pushed onto the supplied unread buffer.</p>

	@param in the input source
	@param urb the cumulative unread buffer
	@return the parsed result (decoded)
	@exception IOException if an I/O error occurs

	@since 2000.01.29
	**********************************************************************/

	private static String tryDoubleQuotedSegment (UnreadBuffer urb) throws IOException
	{
		String raw = tryQuotedCString (urb . getReader (), false);
		if (raw == null)
			return null;
		urb . push (raw);
		return StringToolbox.replace (raw, "\\\n", "");
	}



	//////////////////////////////
	//                          //
	//  character class string  //
	//                          //
	//////////////////////////////




	/**********************************************************************
	Expects class string.

	<p><b>Details:</b>  <code>expectClassString</code> is functionally identical to <code>tryClassString</code>, except that <code>expectClassString</code> throws a <code>StreamFormatException</code> if the parsing operation fails.</p>

	@param in the input source
	@param cl the character class
	@return the class string
	@exception IOException if an I/O error occurs

	@since 2000.06.03
	**********************************************************************/

	public static String expectClassString (PushbackReader in, ICharacterClass cl) throws IOException
	{
		String s = tryClassString (in, cl);
		if (s == null)
			throw new StreamFormatException ("expected character class string");
		return s;
	}



	/**********************************************************************
	Parses class string.

	<p><b>Details:</b>  tryClassString parses a "character class string" from the stream.  A character class string is a string consisting only of characters in the given character class.</p>

	@param in the input source
	@param cl the character class
	@return the class string
	@exception IOException if an I/O error occurs

	@since 2000.06.03
	**********************************************************************/

	public static String tryClassString (PushbackReader in, ICharacterClass cl) throws IOException
	{
		StringBuffer buff = new StringBuffer ();
		while (true)
		{
			int c = in . read ();
			if (c < 0)
				break;
			if (! cl . isMember (c))
			{
				in . unread (c);
				break;
			}
			buff . append ((char) c);
		}
		if (buff . length () == 0)
			return null;
		return buff . toString ();
	}



	/////////////////////////
	//                     //
	//  whitespace string  //
	//                     //
	/////////////////////////




	/**********************************************************************
	Expects white string.

	<p><b>Details:</b>  <code>expectWhiteString</code> is functionally identical to <code>tryWhiteString</code>, except that <code>expectWhiteString</code> throws a <code>StreamFormatException</code> if the parsing operation fails.</p>

	@param in the input source
	@return the white string
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static String expectWhiteString (PushbackReader in) throws IOException
	{
		String s = tryWhiteString (in);
		if (s == null)
			throw new StreamFormatException ("expected whitespace");
		return s;
	}



	private static ICharacterClass white_class = new ICharacterClass ()
	{
		public boolean isMember (int c)
		{
			return Character.isWhitespace ((char) c);
		}
	};



	/**********************************************************************
	Parses white space.

	<p><b>Details:</b>  tryWhiteString parses a "white string" from the stream.  A white string is a string consisting only of characters for which <code>Character.isWhitespace</code> returns <code>true</code>.</p>

	@param in the input source
	@return the white string
	@exception IOException if an I/O error occurs

	@since 2000.01.30
	**********************************************************************/

	public static String tryWhiteString (PushbackReader in) throws IOException
	{
		return tryClassString
		(
			in,
			white_class
		);
	}



	////////////////////////////////////
	//                                //
	//  horizontal whitespace string  //
	//                                //
	////////////////////////////////////




	/**********************************************************************
	Expects horizontal white string.

	<p><b>Details:</b>  <code>expectHorizontalWhiteString</code> is functionally identical to <code>tryHorizontalWhiteString</code>, except that <code>expectHorizontalWhiteString</code> throws a <code>StreamFormatException</code> if the parsing operation fails.</p>

	@param in the input source
	@return the horizontal white string
	@exception IOException if an I/O error occurs

	@since 2000.06.03
	**********************************************************************/

	public static String expectHorizontalWhiteString (PushbackReader in) throws IOException
	{
		String s = tryHorizontalWhiteString (in);
		if (s == null)
			throw new StreamFormatException ("expected whitespace");
		return s;
	}



	private static ICharacterClass horizontal_white_class = new ICharacterClass ()
	{
		public boolean isMember (int c)
		{
			return Ctype.ishspace (c);
		}
	};



	/**********************************************************************
	Parses horizontal white space.

	<p><b>Details:</b>  tryHorizontalWhiteString parses a "horizontal white string" from the stream.  A horizontal white string is a string consisting only of characters for which <code>Ctype.ishspace</code> returns <code>true</code>.</p>

	@param in the input source
	@return the horizontal white string
	@exception IOException if an I/O error occurs

	@since 2000.06.03
	**********************************************************************/

	public static String tryHorizontalWhiteString (PushbackReader in) throws IOException
	{
		return tryClassString
		(
			in,
			horizontal_white_class
		);
	}



	/////////////////////
	//                 //
	//  digits string  //
	//                 //
	/////////////////////



	/**********************************************************************
	Expects digits string.

	<p><b>Details:</b>  <code>expectDigitsString</code> is functionally identical to <code>tryDigitsString</code>, except that <code>expectDigitsString</code> throws a <code>StreamFormatException</code> if the parsing operation fails.</p>

	@param in the stream
	@return the digits string
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static String expectDigitsString (PushbackReader in, int radix) throws IOException
	{
		String s = tryDigitsString (in, radix);
		if (s == null)
			throw new StreamFormatException ("expected digits string");
		return s;
	}



	/**********************************************************************
	Parses digit string.

	<p><b>Details:</b>  parseDigitsString reads a contiguous sequence of digits from the given stream (in).  As many digits as possible are read and consumed, until the first non-<wbr>digit character is encountered.  Only characters that fall below the given radix (<var>radix</var>) are considered digits.  <code>tryDigitsString</code> does <em>not</em> accept or consume sign characters, the radix point, or base-<wbr>specific prefixes such as "0x".  If the first character in the stream does not qualify as a digit, <code>tryDigitsString</code> returns <code>null</code>.</p>

	@param in the stream
	@return the digits string
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public static final String tryDigitsString (PushbackReader in, final int radix) throws IOException
	{
		return tryClassString
		(
			in,
			new ICharacterClass ()
			{
				public boolean isMember (int c)
				{
					return Character.digit ((char) c, radix) >= 0;
				}
			}
		);
	}



	//////////////////////
	//                  //
	//  integer string  //
	//                  //
	//////////////////////



	/**********************************************************************
	Expects integer string.

	<p><b>Details:</b>  <code>expectIntegerString</code> is functionally identical to <code>tryIntegerString</code>, except that <code>expectIntegerString</code> throws a <code>StreamFormatException</code> if the parsing operation fails.</p>

	@param in the input source
	@return the parsed integer string
	@exception IOException if an I/O error occurs

	@since 2000.02.21
	**********************************************************************/

	public static String expectIntegerString (PushbackReader in) throws IOException
	{
		String s = tryIntegerString (in);
		if (s == null)
			throw new StreamFormatException ("expected integer string");
		return s;
	}



	/**********************************************************************
	Parses integer string.

	<p><b>Details:</b>  tryIntegerString attempts to parse an "integer string" from the stream.  An integer string is a numeric string in the following form:</p>

	<ul>
	 <li>99999... (decimal)
	 <li>-99999... (decimal)
	 <li>+99999... (decimal)
	 <li>0xfffff... (hexadecimal)
	 <li>-0xfffff... (hexadecimal)
	 <li>+0xfffff... (hexadecimal)
	 <li>0
	 <li>077777... (octal)
	 <li>-077777... (octal)
	 <li>+077777... (octal)
	</ul>

	<p>The longest string possible that matches any of these forms is consumed from the stream.  (The radix is automatically selected.)  If none of these forms can be parsed, <code>tryIntegerString</code> returns null.  Otherwise, it returns the numeric string that it parsed.</p>

	@param in the input source
	@return the parsed integer string
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public static String tryIntegerString (PushbackReader in) throws IOException
	{
		final int ANY = 0;
		final int SIGNED = 1;
		final int OCT_OR_HEX = 2;
		final int DEC_DONE = 3;
		final int OCT_DONE = 4;
		final int HEX_ONLY = 5;
		final int HEX_DONE = 6;
		StringBuffer buff = new StringBuffer ();
		int state = ANY;
		int c = -1;
	read_loop:
		while (true)
		{
			c = in . read ();
			switch (state)
			{
			case ANY:
				if (c == '+' || c == '-')
					state = SIGNED;
				else if (c == '0')
					state = OCT_OR_HEX;
				else if ('1' <= c && c <= '9')
					state = DEC_DONE;
				else
					break read_loop;
				break;
			case SIGNED:
				if (c == '0')
					state = OCT_OR_HEX;
				else if ('1' <= c && c <= '9')
					state = DEC_DONE;
				else
					break read_loop;
				break;
			case OCT_OR_HEX:
				if ('0' <= c && c <= '7')
					state = OCT_DONE;
				else if (c == 'x' || c == 'X')
					state = HEX_ONLY;
				else
					break read_loop;
				break;
			case DEC_DONE:
				if (c < '0' || '9' < c)
					break read_loop;
				break;
			case OCT_DONE:
				if (c < '0' || '7' < c)
					break read_loop;
				break;
			case HEX_ONLY:
				if (Character.forDigit (c, 16) < 0)
					break read_loop;
				state = HEX_DONE;
				break;
			case HEX_DONE:
				if (Character.forDigit (c, 16) < 0)
					break read_loop;
				break;
			default:
				throw new lava . UnreachableCodeException ();
			}
			buff . append ((char) c);
		}
		in . unread (c);
		switch (state)
		{
		case HEX_ONLY:
			int length = buff . length () - 1;
			in . unread (buff . charAt (length));
			buff . setLength (length);
			// fall through
		case OCT_OR_HEX:
		case DEC_DONE:
		case OCT_DONE:
		case HEX_DONE:
			return buff . toString ();
		case SIGNED:
			in . unread (buff . toString () . toCharArray ());
			// fall through
		case ANY:
			return null;
		default:
			throw new lava.UnreachableCodeException ();
		}
	}



	////////////////////
	//                //
	//  exact string  //
	//                //
	////////////////////



	public static void expectExactString (PushbackReader in, String s) throws IOException
	{
		if (! tryExactString (in, s))
			throw new StreamFormatException ("expected " + s);
	}



	public static boolean tryExactString (PushbackReader in, String s) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
		int length = s . length ();
		for (int i = 0; i < length; ++ i)
		{
			int c = ub . read ();
			if (c != s . charAt (i))
			{
				ub . unreadAll ();
				return false;
			}
		}
		return true;
	}



	///////////////////////
	//                   //
	//  Java identifier  //
	//                   //
	///////////////////////



	/**********************************************************************
	Expects Java identifier.

	<p><b>Details:</b>  expectJavaIdentifier does the same thing as tryJavaIdentifier but throws a StreamFormatException if the parsing is unsuccessful.</p>

	@param in the input source
	@return the Java identifier
	@exception IOException if an I/O error occurs

	@since 2000.06.01
	**********************************************************************/

	public static String expectJavaIdentifier (PushbackReader in) throws IOException
	{
		String ident = tryJavaIdentifier (in);
		if (ident == null)
			throw new StreamFormatException ("expected Java identifier");
		return ident;
	}



	/**********************************************************************
	Parses Java identifier.

	<p><b>Details:</b>  <code>tryJavaIdentifier</code> parses and returns a valid Java identifier.  If a valid Java identifier cannot be parsed, <code>null</code> is returned.</p>

	@param in the input source
	@return the Java identifier
	@exception IOException if an I/O error occurs

	@since 2000.06.01
	**********************************************************************/

	public static String tryJavaIdentifier (PushbackReader in) throws IOException
	{
		boolean started = false;
		StringBuffer ident = new StringBuffer ();
		while (true)
		{
			int c = in . read ();
			if (c < 0)
				break;
			if (started)
				if (Character.isJavaIdentifierPart ((char) c))
					ident . append ((char) c);
				else
				{
					in . unread (c);
					return ident . toString ();
				}
			else
				if (Character.isJavaIdentifierStart ((char) c))
				{
					ident . append ((char) c);
					started = true;
				}
				else
				{
					in . unread (c);
					return null;
				}
		}
		if (started)
			return ident . toString ();
		return null;
	}



	///////////////////////
	//                   //
	//  HTML identifier  //
	//                   //
	///////////////////////



	/**********************************************************************
	Expects HTML identifier.

	<p><b>Details:</b>  expectHtmlIdentifier does the same thing as tryHtmlIdentifier but throws a StreamFormatException if the parsing is unsuccessful.</p>

	@param in the input source
	@return the HTML identifier
	@exception IOException if an I/O error occurs

	@since 2000.07.07
	**********************************************************************/

	public static String expectHtmlIdentifier (PushbackReader in) throws IOException
	{
		String ident = tryHtmlIdentifier (in);
		if (ident == null)
			throw new StreamFormatException ("expected HTML identifier");
		return ident;
	}



	/**********************************************************************
	Parses HTML identifier.

	<p><b>Details:</b>  <code>tryHtmlIdentifier</code> parses and returns a valid HTML identifier.  If a valid HTML identifier cannot be parsed, <code>null</code> is returned.</p>

	<p>An HTML identifier is any string beginning with an upper or lower case letter (A-Z) that consists only of letters, numbers, '_', and '-'.</p>

	@param in the input source
	@return the HTML identifier
	@exception IOException if an I/O error occurs

	@since 2000.07.07
	**********************************************************************/

	public static String tryHtmlIdentifier (PushbackReader in) throws IOException
	{
		boolean started = false;
		StringBuffer ident = new StringBuffer ();
		while (true)
		{
			int c = in . read ();
			if (c < 0)
				break;
			if (started)
				if (isHtmlIdentifierPart (c))
					ident . append ((char) c);
				else
				{
					in . unread (c);
					return ident . toString ();
				}
			else
				if (isHtmlIdentifierStart (c))
				{
					ident . append ((char) c);
					started = true;
				}
				else
				{
					in . unread (c);
					return null;
				}
		}
		if (started)
			return ident . toString ();
		return null;
	}



	// since 2000.07.07
	private static boolean isHtmlIdentifierStart (int c)
	{
		return Ctype.isalpha (c);
	}



	// since 2000.07.07
	private static boolean isHtmlIdentifierPart (int c)
	{
		return Ctype.isalnum (c) || c == '_' || c == '-';
	}



	///////////////////
	//               //
	//  real string  //
	//               //
	///////////////////



	/**********************************************************************
	Expects real number string.

	<p><b>Details:</b>  expectRealString does the same thing as tryRealString but throws a StreamFormatException if the parsing is unsuccessful.</p>

	@param in the input source
	@return the real number string
	@exception IOException if an I/O error occurs

	@since 2000.06.01
	**********************************************************************/

	public static String expectRealString (PushbackReader in) throws IOException
	{
		String rs = tryRealString (in);
		if (rs == null)
			throw new StreamFormatException ("expected real number string");
		return rs;
	}



	/**********************************************************************
	Reads a real number string.

	<p><b>Details:</b>  <code>tryRealString</code> parses and returns a real number string.  The string may be in the form +123.456e+789 or any sensible variation of that.</p>

	@param in the input source
	@return the real number string
	@exception IOException if an I/O error occurs

	@since 2000.06.01
	**********************************************************************/

	public static final String tryRealString (PushbackReader in) throws IOException
	{
		// Implementation based on ParsingReader.readRealString.
		final int NOTHING = 0;
		final int SIGN = 1;
		final int INTEGER = 2;
		final int POINT_NODIGIT = 3;
		final int POINT_DIGIT = 4;
		final int EXPMARK = 5;
		final int EXPSIGN = 6;
		final int EXPDIGIT = 7;
		StringBuffer buff = new StringBuffer ();
		int state = NOTHING;
		int c;
		while (true)
		{
			c = in . read ();
			switch (state)
			{
			case NOTHING:
				if (c == '-' || c == '+')
					state = SIGN;
				else if (Ctype.isdigit (c))
					state = INTEGER;
				else if (c == '.')
					state = POINT_NODIGIT;
				else
				{
					if (c >= 0)
						in . unread (c);
					return null;
				}
				break;
			case SIGN:
				if (Ctype.isdigit (c))
					state = INTEGER;
				else if (c == '.')
					state = POINT_NODIGIT;
				else
				{
					if (c >= 0)
						in . unread (c);
					IoToolbox.unread (in, buff . toString ());
					return null;
				}
				break;
			case INTEGER:
				if (Ctype.isdigit (c))
					break;
				if (c == '.')
					state = POINT_DIGIT;
				else if (c == 'e' || c == 'E')
					state = EXPMARK;
				else
				{
					if (c >= 0)
						in . unread (c);
					return buff . toString ();
				}
				break;
			case POINT_NODIGIT:
				if (Ctype . isdigit (c))
					state = POINT_DIGIT;
				else
				{
					if (c >= 0)
						in . unread (c);
					IoToolbox.unread (in, buff . toString ());
					return null;
				}
				break;
			case POINT_DIGIT:
				if (Ctype . isdigit (c))
					break;
				if (c == 'e' || c == 'E')
					state = EXPMARK;
				else
				{
					if (c >= 0)
						in . unread (c);
					return buff . toString ();
				}
				break;
			case EXPMARK:
				if (Ctype . isdigit (c))
					state = EXPDIGIT;
				else if (c == '+' || c == '-')
					state = EXPSIGN;
				else
				{
					if (c >= 0)
						in . unread (c);
					int length = buff . length () - 1;
					in . unread (buff . charAt (length));
					buff . setLength (length);
					return buff . toString ();
				}
				break;
			case EXPSIGN:
				if (Ctype . isdigit (c))
					state = EXPDIGIT;
				else
				{
					if (c >= 0)
						in . unread (c);
					int length = buff . length ();
					in . unread (buff . charAt (-- length));
					in . unread (buff . charAt (-- length));
					buff . setLength (length);
					return buff . toString ();
				}
				break;
			case EXPDIGIT:
				if (! Ctype.isdigit (c))
				{
					if (c >= 0)
						in . unread (c);
					return buff . toString ();
				}
				break;
			default:
				throw new lava . UnreachableCodeException ();
			}
			buff . append ((char) c);
		}
	}



	/////////////////////
	//                 //
	//  email address  //
	//                 //
	/////////////////////



	/**********************************************************************
	Parses email address.

	<p><b>Details:</b>  tryEmailAddress parses and returns an email address from the stream.  Email addresses parsed by this function are expected to have the following format (using a loose regular expression syntax):</p>

	<p>[a-zA-Z][a-zA-Z0-9\-\.]*@[a-zA-Z0-9]+(\.[a-zA-Z0-9]+)*</p>

	@param in the input source
	@return the email address
	@exception IOException if an I/O error occurs

	@since 2000.09.27
	**********************************************************************/

	public static final String tryEmailAddress (PushbackReader in) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
	parsing:
		{
			String s = tryUserName (in);
			if (s == null)
				break parsing;
			ub . push (s);
			if (ub . read () != '@')
				break parsing;
			s = tryHostName (in);
			if (s == null)
				break parsing;
			ub . push (s);
			return ub . getContents ();
		}
		ub . unreadAll ();
		return null;
	}



	private static final String tryUserName (PushbackReader in) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
	parsing:
		{
			if (! isUsernameStart (ub . read ()))
				break parsing;
			while (isUsernamePart (ub . read ()))
				continue;
			ub . unreadEos ();
			return ub . getContents ();
		}
		ub . unreadAll ();
		return null;
	}



	private static boolean isUsernameStart (int c)
	{
		return Ctype.isalpha (c);
	}



	private static boolean isUsernamePart (int c)
	{
		return
			Ctype.isalpha (c)
		||	Ctype.isdigit (c)
		||	c == '-'
		||	c == '.';
	}



	private static String tryHostName (PushbackReader in) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
	parsing:
		{
			String s = tryZone (in);
			if (s == null)
				break parsing;
			ub . push (s);
			while (true)
			{
				if (ub . read () != '.')
				{
					ub . unreadEos ();
					break;
				}
				s = tryZone (in);
				if (s == null)
				{
					// unread '.'
					ub . unread ();
					break;
				}
				ub . push (s);
			}
			return ub . getContents ();
		}
		ub . unreadAll ();
		return null;
	}



	private static String tryZone (PushbackReader in) throws IOException
	{
		UnreadBuffer ub = new UnreadBuffer (in);
		parsing:
		{
			if (! isZoneStart (ub . read ()))
				break parsing;
			while (isZonePart (ub . read ()))
				continue;
			ub . unreadEos ();
			if (! isZoneEnd (ub . peekPop ()))
				break parsing;
			return ub . getContents ();
		}
		ub . unreadAll ();
		return null;
	}



	private static boolean isZoneStart (int c)
	{
		return
			Ctype.isalpha (c)
		||	Ctype.isdigit (c);
	}



	private static boolean isZonePart (int c)
	{
		return
			Ctype.isalpha (c)
		||	Ctype.isdigit (c)
		||	c == '-';
	}



	private static boolean isZoneEnd (int c)
	{
		return isZoneStart (c);
	}



	/////////////////////
	//                 //
	//  miscellaneous  //
	//                 //
	/////////////////////



	/**********************************************************************
	Searches for string.

	<p><b>Details:</b>  <code>searchString</code> consumes characters from the given stream (<var>in</var>) until the given string (<var>target</var>) is found, the search limit (<var>limit</var>) is exceeded, or the stream runs out.  The search limit is the number of characters within which the first character of the target must be observed.</p>

	<p>If the target string is found, and a non-<code>null</code> <code>StringBuffer</code> is supplied, then all of the characters that were skipped, including the target, will be written into (appended to) the supplied <code>StringBuffer</code> (<var>skipped_text</var>).  If the <var>target</var> is not found, and a <code>StringBuffer</code> was supplied, then it will contain the characters that were read before the search was aborted.</p>

	<p><code>true</code> is returned if the string is found, <code>false</code> otherwise.</p>

	<p>Note that setting <var>limit</var> to 0 results in behavior similar to tryExactString.</p>

	@param in the input source
	@param target the string to find and skip
	@param limit how far ahead to search
	@param skipped_text the skipped characters
	@return true iff target is found
	@exception IOException if an I/O error occurs

	@since 2000.06.29
	**********************************************************************/

	public static boolean searchString
	(
		Reader in,
		String target,
		int limit,
		StringBuffer skipped_text
	)
		throws IOException
	{
		// Trivial case:
		int target_len = target . length ();
		if (target_len == 0)
			return true;
		// Use our own buffer if one was not supplied:
		StringBuffer memory;
		if (skipped_text != null)
			memory = skipped_text;
		else
			memory = new StringBuffer ();
		// Scan for matches:
		int matched = 0;
	scan:
		for (int consumed = 0; consumed - matched < limit; ++ consumed)
		{
			int c = in . read ();
			if (c < 0)
				return false;
			memory . append ((char) c);
			if (c == target . charAt (matched))
				++ matched;
			else
			{
				if (matched > 0)
				{
					++ matched;
					int match_start = memory . length () - matched ;
					String tail = memory . substring (match_start);
					for (int i = 1; i < matched; ++ i)
						if (tail . regionMatches (i, target, 0, matched - i))
						{
							matched = matched - i;
							continue scan;
						}
					matched = 0;
					// Conserve memory if there is no need to keep the consumed characters:
					if (skipped_text != memory)
						memory . setLength (0);
				}
				continue scan;
			}
			if (matched == target_len)
				return true;
		}
		return false;
	}



	/**********************************************************************
	Creates line parser.

	<p><b>Details:</b>  <code>getLineParser</code> returns a <code>PushbackReader</code> that reads characters from the given string (<var>line</var>) and which has enough pushback buffer to push back the entire string if necessary.</p>

	@param line the line to parse
	@return the PushbackReader

	@since 2000.11.13
	**********************************************************************/

	public static PushbackReader getLineParser (String line)
	{
		return new PushbackReader
		(
			new StringReader (line),
			line . length ()
		);
	}



} // class StreamParser



