///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.File;
import java.io.IOException;
import java.io.FileWriter;



/*****************************************************************************
Writes data to temporary file.

<p><b>Details:</b> A <code>TemporaryFileWriter</code> is a specialized <code>FileWriter</code> that automatically stores its output in a temporary file.  This class is useful for situation where data must be saved, but the choice of filename is inconsequential.  <code>TemporaryFileWriter</code> always attempts to use a filename that is not already in use.</p>

<p>After a <code>TemporaryFileWriter</code> has been closed, the temporary file can be renamed, possibly overwriting the contents of a destination file.  Thus, this class is ideal for applications which must read and rewrite files.</p>

<p><small><b>Note:</b> Don't forget that because this class is a subclass of <code>FileWriter</code>, all of the superclass' methods are available in this class.</small></p>

<p><b>Note:</b> It appears the people at Javasoft liked this idea.  Similar features have now been incorporated into the 1.2 release of Java.  (Just remember where it appeared first! :-)  This class remains a part of Lava, however, for backword compatibility with existing programs and also to provide continued support for JDK 1.1.  Even 1.2 users may find this class to be a more flexible solution.</p>

@author Sharky
*****************************************************************************/

public class TemporaryFileWriter extends FileWriter
{



	/*********************************************************************
	The temporary file.
	*********************************************************************/

	private File file;



	/*********************************************************************
	Creates an instance that writes to the specified temporary file.
	*********************************************************************/

	private TemporaryFileWriter (Object _null, File file) throws IOException
	{
		super (file);
		this . file = file;
	}




	private File rename_dest;



	/*********************************************************************
	Creates an instance that writes to a randomly chosen temporary file.
	*********************************************************************/

	public TemporaryFileWriter () throws IOException
	{
		this (null, TemporaryFile.choose_file (new File (".")));
		rename_dest = null;
	}




	/*********************************************************************
	Creates an instance that writes to a randomly chosen temporary file in the same directory as the given file.  If rename_dest is a file, the temporary file will be renamed to that location when it is closed, and the destination, if it exists, will be overwritten.  If the given file is a directory, however, then <code>rename_dest</code> will only be used to determine which directory to store the temporary file in, and the caller will be responsible for eventually renaming (or deleting) the file on his own.
	*********************************************************************/

	public TemporaryFileWriter (File rename_dest) throws IOException
	{
		this (null, TemporaryFile.choose_file (rename_dest));
		if (! rename_dest . isDirectory ())
			this . rename_dest = rename_dest;
	}




	/**********************************************************************
	Indicates whether the file is still open.  This variable is <code>true</code> until one of the close methods has been called.
	**********************************************************************/

	private boolean open = true;



	/*********************************************************************
	Closes the output stream and optionally renames the temporary file.  If the destination file already exists, it will be replaced.  If no destination file is specified, then the temporary file will not be renamed.  In either case, a <code>File</code> object representing the completed temporary file will be returned.  It is OK to call this method after the <code>close</code> method, but it is not necessary.</p>

	<p>Tip: It may not be possible to rename the temporary file if the destination file is in use.  If you are using this class to rewrite a file, make sure you close the source before you close this destination, so that the source file will not be considered "in use."

	@param new_file a File object representing the new filename for the completed temporary file, or null if the file should not be renamed
	@return a File object representing the completed file
	*********************************************************************/

	public File closeAndRename (File new_file) throws IOException
	{
		super . close ();
		open = false;
		if (new_file == null)
			return file;
		if (new_file . exists ())
		{
			new_file . delete ();
			if (new_file . exists ())
				throw new IOException ("can't delete destination file " + new_file);
		}
		file . renameTo (new_file);
		file = new_file;
		return new_file;
	}



	/**********************************************************************
	Closes the temporary file and, if a rename destination was specified when this instance was constructed, attempts to rename it.  If a file exists by the same name as the rename destination, it will be overwritten.

	@exception IOException if an I/O error occurs
	**********************************************************************/

	public void close () throws IOException
	{
		closeAndRename (rename_dest);
		rename_dest = null;
	}



	/**********************************************************************
	Returns a <code>File</code> object representing this stream's temporary file.  This method can only be called after the close or closeAndRename method has been called.

	@return the temporary file
	**********************************************************************/

	public final File getFile () throws IOException
	{
		if (open)
			throw new IOException ("getFile was called before the file was closed");
		return file;
	}



}



