///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.BufferedReader;
import java.io.IOException;
import java.io.LineNumberReader;



/******************************************************************************
Filters any-platform text stream to appear as if it originated from a Unix file.

<p>Don't let the name fool you!  <code>UnixLineReader</code> is useful on all platforms.  It allows you to read a text stream without worrying about which platform the stream was created on.  Unix, Macintosh, and PCs all create text files using different endline sequences.  Unfortunately, using system-independent code to get the <code>line.separator</code> property is an inadequate solution to the problem, since a program running on one platform may be reading a text file that was created on another platform.</p>

<p>This class automatically recognizes any of the three major line-ending sequences (Unix: "\n"; Win32: "\r\n"; Macintosh: "\r") -- even if they are intermixed within the stream (i.e., by concatenating files from different platforms) -- and transparently converts the line separators to the standard Unix endline character '\n' when they are read.</p>

<p>Knowing what kind of endline sequence to expect greatly simplifies the process of parsing line ends when you are reading the stream character by character.  Looking for the end of the line?  Just look for the '\n' character, and don't worry about trying to guess which platform the stream was created on!  This filter also makes sure the last line is terminated with a line separator, even it this was not the case in the original source.</p>

<p><b>Skip characters.</b>  The counts of characters skipped are given in terms of <em>filtered</em> characters.  (This matters only when reading DOS or Windows text files, since the endline sequence is 2 characters, but counted only as 1 character by this method.)  The number of characters actually skipped is returned.  A zero return value may indicate EOF.

<!--SHOWSOURCE-->
******************************************************************************/

public class UnixLineReader
	extends LFilterReader
{



	// This same as in, but pre-casted as a BufferedReader for convenience and speed.
	private transient BufferedReader bin;



	private String cur_line = null;
	private int cur_line_pos = 0;
	private String marktime_cur_line = null;
	private int marktime_cur_line_pos = 0;



	private boolean prep_line () throws IOException
	{
		if (cur_line != null)
			return true;
		cur_line = bin . readLine ();
		if (cur_line == null)
			return false;
		cur_line += "\n";
		cur_line_pos = 0;
		return true;
	}



	/**********************************************************************
	Initializes a new instance to read from the specified reader.

	@param reader the reader
	**********************************************************************/

	public UnixLineReader (BufferedReader reader)
	{
		super (reader);
		bin = reader;
	}



	/**********************************************************************
	Reads the next character from the stream.

	@return the character read
	**********************************************************************/

	public int read () throws IOException
	{
		return _read ();
	}



	/**********************************************************************
	Does the work for read().  This method is installed to prevent unexpected polymorphism problems.
	**********************************************************************/

	final int _read () throws IOException
	{
		if (! prep_line ())
			return -1;
		int c = cur_line . charAt (cur_line_pos ++);
		if (cur_line_pos == cur_line . length ())
			cur_line = null;
		return c;
	}



	/**********************************************************************
	Reads a block of characters from the stream.

	@param dest destination array for storing the characters read
	@param off starting offset into the destination array
	@param len number of characters to read
	@return number of characters actually read (may be lest than len, indicating EOF or IO trouble)
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public int read (char [] dest, int off, int len) throws IOException
	{
		return read_by_singles (dest, off, len);
	}



	int read_by_singles (char [] dest, int off, int len) throws IOException
	{
		int i;
		for (i = 0; i < len; ++ i)
		{
			int c = _read ();
			if (c < 0)
				break;
			dest [off + i] = (char) c;
		}
		if (i == 0)
			i = len <= 0 ? 0 : -1;
		return i;
	}



	/**********************************************************************
	Mark the present position in this filtered stream.  Subsequent calls to reset() will attempt to reposition the stream to the position and state it was in when this method was called.  There is a limit to how many characters can be read ahead before attempts to reset will no longer succeed.  Not all streams support mark().  Subclasses that override this method need to call <code>super.mark</code>.

	@param read_ahead_limit the read ahead limit
	@exception IOException if the stream does not support mark()
	**********************************************************************/

	public void mark (int read_ahead_limit) throws IOException
	{
		in . mark (read_ahead_limit);
		marktime_cur_line = cur_line;
		marktime_cur_line_pos = cur_line_pos;
	}



	/**********************************************************************
	Attempts to reset the stream.  If the stream has been marked, then this method will attempt to store the state of the stream at the time it was marked.  If the stream has not been marked, then this method will attempt to reset the stream in some way that is appropriate to this particular stream.  Not all streams support reset().  Subclasses that override this method need to call <code>super.reset</code>.

	@exception IOException if the stream cannot be reset
	**********************************************************************/

	public void reset () throws IOException
	{
		in . reset ();
		cur_line = marktime_cur_line;
		cur_line_pos = marktime_cur_line_pos;
	}



	/**********************************************************************
	Reads a line, or the rest of the current line if some of the current line has already been read.

	@return the line that was read, or null if EOF
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public final String readLine () throws IOException
	{
		String result;
		if (cur_line == null)
			result = bin . readLine ();
		else
		{
			result = cur_line . substring (cur_line_pos, cur_line . length () - 1);
			cur_line = null;
		}
		return result;
	}



	/**********************************************************************
	Returns the total number of lines read so far.  This method works only if the underlying reader, provided at construction time, is an instance of LineNumberReader.  Otherwise, a ClassCastException will be thrown.

	@return the total number of lines read
	@exception ClassCastException if the base reader is not a LineNumberReader
	**********************************************************************/

	public int getLineNumber ()
	{
		if (cur_line == null)
			return ((LineNumberReader) bin) . getLineNumber ();
		else
			return ((LineNumberReader) bin) . getLineNumber () - 1;
	}



	/**********************************************************************
	Sets the current line number.  This method works only if the underlying reader, provided at construction time, is an instance of LineNumberReader.  Otherwise, a RuntimeException will be thrown.

	@return the total number of lines read
	@exception ClassCastException if the base reader is not a LineNumberReader
	**********************************************************************/

	public void setLineNumber (int n)
	{
		try
		{
			if (cur_line == null)
				((LineNumberReader) bin) . setLineNumber (n);
			else
				((LineNumberReader) bin) . setLineNumber (n + 1);
		}
		catch (ClassCastException e)
		{
			throw new RuntimeException ("base reader does not support line numbering");
		}
	}



}



