///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.IOException;
import java.io.PushbackReader;
import java.io.Reader;
import lava.string.CharacterStack;



/******************************************************************************
"Pushbackable" input stream with no pushback limit.

<p><b>Details:</b>  This class is similar to <code>java.io.PushbackReader</code>, except that no limit is imposed on the number of characters that can be pushed back.  Additionally, the unread methods can process either characters or whole strings.</p>

@since   1998.02.22
@version 2000.06.03
******************************************************************************/

public class UnlimitedPushbackReader
	extends PushbackReader
{



	/*********************************************************************
	The current line buffer.
	*********************************************************************/

	private CharacterStack pushback_chars = new CharacterStack ();



	/*********************************************************************
	The line buffer at the time of marking.
	*********************************************************************/

	private CharacterStack marktime_pushback_chars = new CharacterStack ();



	/**********************************************************************
	Does the work for read().

	<p><b>Details:</b>  _read does all the dirty work for read.  This method is installed to prevent unexpected polymorphism loops.</p>

	<p><b>Called by:</b> read(), super.read_by_singles(char[],int,int)</p>

	@return the character read
	@exception IOException if an I/O error occurs

	@version 1998
	@since 1998
	**********************************************************************/

	private final int _read () throws IOException
	{
		int c = pushback_chars . pop ();
		if (c >= 0)
			return c;
		return in . read ();
	}



	private int read_by_singles (char [] dest, int off, int len) throws IOException
	{
		int i;
		for (i = 0; i < len; ++ i)
		{
			int c = _read ();
			if (c < 0)
				break;
			dest [off + i] = (char) c;
		}
		if (i == 0)
			i = len <= 0 ? 0 : -1;
		return i;
	}



	/**********************************************************************
	Converts a <code>Reader</code> to an <code>UnlimitedPushbackReader</code>.

	@param reader the <code>Reader</code>

	@version 1998
	@since 1998
	**********************************************************************/

	public UnlimitedPushbackReader (Reader reader)
	{
		super (reader);
	}



	/**********************************************************************
	Reads the next character.

	<p><b>Details:</b>  <code>read</code> reads the next character from the stream and returns it.  If the end-of-stream has been reached, -1 is returned.</p>

	@return the character read
	@exception IOException if an I/O error occurs

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public int read () throws IOException
	{
		return _read ();
	}



	public final int read (char[] dest) throws IOException
	{
		return read (dest, 0, dest . length);
	}



	/**********************************************************************
	Reads a block of characters.

	<p><b>Details:</b>  This method reads a block of characters from this stream, filling <code>dest</code>, the supplied destination buffer.  <code>len</code> characters are written into <code>dest</code> begining at offset <code>off</code>.  If no characters can be read before encountering the end-of-stream, -1 is returned.  Otherwise, the number of characters read and stored in <code>buff</code> is returned.  Note that if end-of-stream is reached while filling the buffer, this value may be less than <code>len</code>.</p>

	@param dest the destination buffer
	@param off offset into dest
	@param len number of characters to read
	@return number of characters read and stored
	@exception IOException if an I/O error occurs

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public int read (char [] dest, int off, int len) throws IOException
	{
		return read_by_singles (dest, off, len);
	}



	/**********************************************************************
	Previews the next character.

	<p><b>Details:</b>  <code>peek</code> <code>read</code>s and <code>unread</code>s the next character, and then returns its value.</p>

	<p><b>Calls:</b> <code>read()</code>, <code>unread(int)</code>

	@return the character read
	@exception IOException if an I/O error occurs

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public int peek () throws IOException
	{
		int c = read ();
		unread (c);
		return c;
	}



	/**********************************************************************
	Pushes a character back to the stream.

	<p><b>Details:</b>  <code>unread</code> pushes a character back onto the stream.  Subsequent calls to <code>read</code> will retrieve it.  Unreading an EOF is allowed and has no effect.</p>

	@param c the character

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public void unread (int c) throws IOException
	{
		if (c < 0)
			return;
		pushback_chars . push ((char) c);
	}



	public final void unread (char[] src) throws IOException
	{
		unread (src, 0, src . length);
	}



	public void unread (char [] src, int off, int len) throws IOException
	{
		for (int i = len - 1; i >= 0; -- i)
			pushback_chars . push (src [off + i]);
	}



	/**********************************************************************
	Pushes a String back to the stream.

	<p><b>Details:</b>  <code>unread</code> pushes a String back onto the stream.  Subsequent calls to <code>read</code> will retrieve its characters.  Unreading <code>null</code> is allowed and has no effect.</p>

	@param s the <code>String</code>

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public void unread (String s) throws IOException
	{
		if (s == null)
			return;
		pushback_chars . push (new StringBuffer (s) . reverse () . toString ());
	}



	/**********************************************************************
	Marks the present position.

	<p><b>Details:</b>  <code>mark</code> marks the present position in this stream.  Subsequent calls to <code>reset</code> will attempt to restore the stream to the state it was at when <code>mark</code> was last called.  There is a limit on how many characters can be read (<code>read_ahead_limit</code>), after calling <code>mark</code>, before calls to <code>reset</code> are no longer guaranteed to be successful.  Not all streams support <code>mark</code>.  This stream supports <code>mark</code> if the base stream supports <code>mark</code>.</p>

	@param read_ahead_limit number of characters that can be read after <code>mark</code> before <code>reset</code>-ability is compromised
	@exception IOException if the stream cannot be marked

	@version 1998
	@since 1998
	**********************************************************************/

	public void mark (int read_ahead_limit) throws IOException
	{
		in . mark (read_ahead_limit);
		marktime_pushback_chars = (CharacterStack) pushback_chars . clone ();
	}



	/**********************************************************************
	Restores this stream to the previously marked state.

	<p><b>Details:</b>  <code>reset</code> attempts to restore the stream to the state it was in during the most recent call to <code>mark</code>.  If <code>mark</code> has not been previously called, then <code>reset</code> will attempt to reset this stream in a manner that is appropriate for this particular type of stream.  Not all streams support <code>reset</code>.  This stream supports <code>reset</code> if the base stream supports <code>reset</code>.</p>

	@exception IOException if the stream cannot be reset

	@since 1998
	@version 1999.12.03
	**********************************************************************/

	public void reset () throws IOException
	{
		in . reset ();
		pushback_chars = (CharacterStack) marktime_pushback_chars . clone ();
	}



	/**********************************************************************
	Tells if the end-of-stream has been reached.

	<p><b>Details:</b>  <code>eof</code> calls <code>peek</code> to see if another character is available in the stream.  If the end-of-stream has been reached, <code>peek</code> returns <code>true</code>, otherwise <code>false</code>.</p>

	<p>Technically, this method checks for end-of-stream, not end-of-file, since the base stream may not originate from a file.  This method is named "eof" for traditional reasons.</p>

	<p><b>Calls:</b> peek()</p>

	@return <code>true</code> iff the end-of-stream has been reached
	@exception IOException if an I/O error occurs

	@version 1998.11.11
	@since 1998
	**********************************************************************/

	public boolean eof () throws IOException
	{
		return peek () < 0;
	}



	public void close () throws IOException
	{
		in . close ();
	}



}



