///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.io;



import java.io.IOException;
import java.io.PushbackReader;



/******************************************************************************
Enhanced unread support for <code>PushbackReader</code>s.

<p><b>Details:</b> <code>UnreadBuffer</code> works in tandem with <code>PushbackReader</code> to provide sophisticated "undo" support while parsing text streams.  An <code>UnreadBuffer</code> can be attached to a <code>PushbackReader</code> at any stage in the parsing process and used to remember the characters while you are parsing them.  This makes it easy to unread entire segments as your parser discovers it has gone down an incorrect grammar path, or to recall data that has been read as you discover that the parsing path was correct.  Essentially, this class eliminates the need for <code>peek</code>, <code>unread</code>, <code>mark</code>, and <code>reset</code> in the client code.</p>

<blockquote>

	<p><b>Example:</b> Here is a function that (greedily) parses strings of letters that begin with '<tt>a</tt>' and end with '<tt>z</tt>', while rejecting everything else.  If a qualifying string is parsed, it is removed from the stream and returned.  If such a string is not next in the stream, however, <code>null</code> is returned and no characters are consumed from the stream (i.e., everything read is pushed back).</p>

	<pre>
	*static String parseAtoZ (PushbackReader pr) throws IOException
	*{
	*	UnreadBuffer ub = new UnreadBuffer (pr);
	*parsing:
	*	{
	*		if (ub . read () != 'a')
	*			break parsing;
	*		while (true)
	*		{
	*			int c = ub . read ();
	*			if (c == 'z')
	*				return ub . getContents ();
	*			if (! Ctype.isalpha (c))
	*				break parsing;
	*		}
	*	}
	*	ub . unreadAll ();
	*	return null;
	*}
	</pre>

</blockquote>

<p>Since the idea of this class is to allow you to do a whole bunch of reading while still retaining your ability to unread, you should make sure that the source input stream is capable of the amount of consecutive unreads that your use of this class may require.  You may also consider using the {@link UnlimitedPushbackReader <code>UnlimitedPushbackReader</code>} so that you don't have to worry about it.</p>

@since 2000.06.01
@author Sharky
******************************************************************************/

public class UnreadBuffer
{



	private final PushbackReader in;



	/**********************************************************************
	Returns source stream.

	<p><b>Details:</b>  <code>getReader</code> returns the source stream that was given in the constructor.  There should be very little need to call this method in well-<wbr>designed code, but it is available as a convenience.</p>

	@return source stream
	**********************************************************************/

	public PushbackReader getReader ()
	{
		return in;
	}



	/**********************************************************************
	Sets source stream.

	<p><b>Details:</b>  This constructor sets the source PushbackReader from which data is read and to which data is unread.</p>

	@param in the source PushbackReader
	**********************************************************************/

	public UnreadBuffer (PushbackReader in)
	{
		this . in = in;
	}



	private StringBuffer buffer = new StringBuffer ();



	private boolean eos_reached = false;



	/**********************************************************************
	Reads single character.

	<p><b>Details:</b>  <code>read</code> reads a single character from the source stream, pushes it onto the internal unread stack in case it needs to be referenced later, and then returns the character.  If the source stream contains no more characters and produces an EOS symbol, that symbol will be returned, but it will <em>not</em> be pushed onto the unread stack.</p>

	@return character read
	@exception IOException if an I/O error occurs
	**********************************************************************/

	public int read () throws IOException
	{
		int c = in . read ();
		push (c);
		eos_reached = c < 0;
		return c;
	}



	/**********************************************************************
	Unreads one character.

	<p><b>Details:</b>  <code>unread</code> pops a character from the unread stack and pushes it back onto the source stream.  If there are no characters on the stack to pop, this method does nothing.  (This makes it safe to call unread after detecting an unexpected character, which may or may not be the EOS symbol.)</p>

	@exception IOException if an I/O error occurs
	**********************************************************************/

	public void unread () throws IOException
	{
		int c = pop ();
		if (c < 0)
			return;
//			throw new IOException ("nothing in buffer to unread");
		in . unread (c);
	}



	/**********************************************************************
	Unreads if not EOS.

	<p><b>Details:</b>  <code>unreadEos</code> performs the same operation as unread, but only if the last call to read did not return the EOS symbol.</p>

	@exception IOException if an I/O error occurs

	@since 2000.09.28
	**********************************************************************/

	public void unreadEos () throws IOException
	{
		if (! eos_reached)
			unread ();
	}



	/**********************************************************************
	Unreads all characters.

	<p><b>Details:</b>  <code>unreadAll</code> pops all the characters from the unread stack and pushes them back onto the source stream.</p>
	**********************************************************************/

	public synchronized void unreadAll () throws IOException
	{
		in . unread (popAll () . toCharArray ());
	}



	/**********************************************************************
	Pushes character onto unread stack.

	<p><b>Details:</b>  <code>push</code> appends a single character (<var>c</var>) to the unread stack.  <var>c</var> is automatically cast to a <code>char</code> first.  If <var>c</var> is negative, this method does nothing.</p>

	@param c character to push
	**********************************************************************/

	public synchronized void push (int c)
	{
		if (c < 0)
			return;
		buffer . append ((char) c);
	}



	/**********************************************************************
	Pushes character onto unread stack.

	<p><b>Details:</b>  <code>push</code> appends a single character (<var>c</var>) to the unread stack.</p>

	@param c character to push
	**********************************************************************/

	public synchronized void push (char c)
	{
		buffer . append (c);
	}



	/**********************************************************************
	Pushes string onto unread stack.

	<p><b>Details:</b>  <code>push</code> appends a string (<var>s</var>) to the unread stack.  The last character in the string is pushed last.</p>

	@param s string to push
	**********************************************************************/

	public synchronized void push (String s)
	{
		if (s == null)
			return;
		buffer . append (s);
	}



	/**********************************************************************
	Removes character from unread stack.

	<p><b>Details:</b>  pop removes a single character from the unread stack and returns it.  If the stack is empty, <code>pop</code> returns -1.</p>

	@return popped character
	**********************************************************************/

	public synchronized int pop ()
	{
		int new_length = buffer . length () - 1;
		if (new_length < 0)
			return -1;
		int c = buffer . charAt (new_length);
		buffer . setLength (new_length);
		return c;
	}



	/**********************************************************************
	Pops entire unread stack.

	<p><b>Details:</b>  <code>popAll</code> pops the entire unread stack and returns it as a string.  If the stack is empty, an empty string is returned.</p>

	@return popped string

	@see #getContents()
	**********************************************************************/

	public synchronized String popAll ()
	{
		String s = buffer . toString ();
		buffer . setLength (0);
		return s;
	}



	/**********************************************************************
	Returns entire unread stack.

	<p><b>Details:</b>  Like <code>popAll</code>, <code>getContents</code> returns the entire contents of the unread stack, but it does so without affecting the stack's contents.</p>

	@return unread stack

	@see #popAll()
	**********************************************************************/

	public synchronized String getContents ()
	{
		return buffer . toString ();
	}



	/**********************************************************************
	Peeks ahead one character.

	<p><b>Details:</b>  Although not very useful and certainly not necessary, peek is available as a convenience.  It does what most <code>peek</code>s do: it allows you to preview the next character in the stream without consuming it and, in this case, without adding it to the unread stack.  In other words, <code>peek</code> returns what <code>read</code> would return if you called it, but without affect the state of this <code>UnreadBuffer</code>.</p>

	@return peeked character
	**********************************************************************/

	public synchronized int peek () throws IOException
	{
		return IoToolbox.peek (in);
	}



	/**********************************************************************
	Peeks top of stack.

	<p><b>Details:</b>  <code>peekPop</code> returns the character on the top of the unread stack.  If the stack is empty, <code>peekPop</code> returns -1.</p>

	@return character on top

	@since 2000.09.28
	**********************************************************************/

	public int peekPop ()
	{
		int length = buffer . length ();
		if (length == 0)
			return -1;
		return buffer . charAt (length - 1);
	}



//	/**********************************************************************
//	Pushes back given character.
//
//	<p><b>Details:</b>  <code>unread</code> passes the given character (<var>c</var>) to the <code>unread</code> method of the underlying <code>PushbackReader</code>, but only if <var>c</var> &gt;= 0.  This means it's safe to push back a character even if it might be the EOS marker.  The character stack in this <code>UnreadBuffer</code> is unaffected.</p>
//
//	@param c character to push back
//	@exception IOException if an I/O error occurs
//
//	@since 2000.09.28
//	**********************************************************************/
//
//	public void unread (int c) throws IOException
//	{
//		if (c >= 0)
//			in . unread (c);
//	}



}



