///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.string;



import java.math.BigInteger;
import lava.konst.BIG_INTEGER;



/******************************************************************************
Converts strings to numbers.

<p><b>Details:</b> NumberString is a collection of functions that perform various conversions from <code>String</code>s into numeric types.</p>

@author Sharky
******************************************************************************/

public class NumberString
{



	/**********************************************************************
	Converts numeric string to BigInteger.

	<p><b>Details:</b>  toBigInteger accepts numeric strings in the following form:</p>

	<ul>
	 <li>99999... (decimal)
	 <li>-99999... (decimal)
	 <li>+99999... (decimal)
	 <li>0xfffff... (hexadecimal)
	 <li>-0xfffff... (hexadecimal)
	 <li>+0xfffff... (hexadecimal)
	 <li>0
	 <li>077777... (octal)
	 <li>-077777... (octal)
	 <li>+077777... (octal)
	</ul>


	<p>and safely converts them to <code>BigInteger</code>s without threat of throwing an exception.  <i>(The <code>BigInteger</code> constructors and methods can't do this.)</i>  If the supplied string is <code>null</code> or not in any of the above forms, <code>null</code> is returned.  <i>(BigInteger can't do this either!)</i>  In no case will an exception be thrown.  <i>(Nor this... :-)</i></p>

	@param s the string
	@return the BigInteger

	@since 1998.11.18
	**********************************************************************/

	public static BigInteger toBigInteger (String s)
	{
		try
		{
			boolean neg = false;
			switch (s . charAt (0))
			{
			case '-':
				neg = true;
				s = s . substring (1);
				break;
			case '+':
				s = s . substring (1);
			}
			int radix = 10;
			if (s . startsWith ("0"))
			{
				if (s . length () == 1)
					return BigInteger.valueOf (0);
				switch (s . charAt (1))
				{
				case 'x':
				case 'X':
					s = s . substring (2);
					radix = 16;
					break;
				default:
					radix = 8;
				}
			}
			// At this point, there should just be pure numbers in the string's intended radix, since we stripped out the sign and radix prefixes.
			if (s . indexOf ('-') >= 0 || s . indexOf ('+') >= 0)
				return null;
			BigInteger bi = new BigInteger (s, radix);
			if (neg)
				bi = bi . negate ();
			return bi;

		}
		catch (NullPointerException e)
		{
		}
		catch (NumberFormatException e)
		{
		}
		catch (StringIndexOutOfBoundsException e)
		{
		}
		return null;
	}



	/**********************************************************************
	Decodes signed long.

	<p><b>Details:</b>  <code>toSignedLong</code> is functionally the same as <code>toBigInteger</code>, except that <code>toSignedLong</code> attempts to decode a <code>long</code> instead of a <code>BigInteger</code>.  Failure occurs if the string does not contain one of the forms recognized by <code>toBigInteger</code> or if the string represents a number that is outside of the range of a signed <code>long</code>.  If the conversion is successful, a <code>Long</code> representing the converted value is returned.  Otherwise, <code>null</code> is returned.</p>

	@param s the string to decode
	@return the decoded value

	@since 2000.02.21
	**********************************************************************/

	public final static Long toSignedLong (String s)
	{
		BigInteger bi = toBigInteger (s);
		if (bi == null)
			return null;
		if
		(
			bi . compareTo (BIG_INTEGER.MAX_SLONG) > 0
		||	bi . compareTo (BIG_INTEGER.MIN_SLONG) < 0
		)
			return null;
		return new Long (bi . longValue ());
	}



	/**********************************************************************
	Decodes signed int.

	<p><b>Details:</b>  <code>toSignedInt</code> is functionally the same as <code>toBigInteger</code>, except that <code>toSignedInt</code> attempts to decode an <code>int</code> instead of a <code>BigInteger</code>.  Failure occurs if the string does not contain one of the forms recognized by <code>toBigInteger</code> or if the string represents a number that is outside of the range of a signed <code>int</code>.  If the conversion is successful, an <code>Integer</code> representing the converted value is returned.  Otherwise, <code>null</code> is returned.</p>

	@param s the string to decode
	@return the decoded value

	@since 2000.02.21
	**********************************************************************/

	public final static Integer toSignedInt (String s)
	{
		BigInteger bi = toBigInteger (s);
		if (bi == null)
			return null;
		if
		(
			bi . compareTo (BIG_INTEGER.MAX_SINT) > 0
		||	bi . compareTo (BIG_INTEGER.MIN_SINT) < 0
		)
			return null;
		return new Integer (bi . intValue ());
	}



	/**********************************************************************
	Safely converts a <code>String</code> into a <code>BigInteger</code>.  <code>null</code> is returned if the string cannot be converted.

	@param s the string to convert
	@param radix the radix for conversion
	@return the BigInteger represented by the string
	**********************************************************************/

	private static BigInteger to_int (String s, int radix)
	{
		if (s == null)
			return null;
		if (s . length () == 0)
			return null;
		// (The constructor for BigInteger does not allow a positive sign prefix.)
		if (s . charAt (0) == '+')
			s = s . substring (1);
		try
		{
			return new BigInteger (s, radix);
		}
		catch (NumberFormatException e)
		{
			return null;
		}
	}




	/**********************************************************************
	Safely converts String to int.

	<p><b>Details:</b>  <code>toSignedInt</code> converts a numeric string to a signed 32-bit integer.</p>

	@param s the string to convert
	@param radix the radix in which the string's numeric value is expressed
	@return an Integer representing the converted value, or null if the conversion failed
	**********************************************************************/

	public final static Integer toSignedInt (String s, int radix)
	{
		BigInteger bi = to_int (s, radix);
		if (bi == null)
			return null;
		if
		(
			bi . compareTo (BIG_INTEGER.MAX_SINT) == 1
		||
			bi . compareTo (BIG_INTEGER.MIN_SINT) == -1
		)
			return null;
		return new Integer (bi . intValue ());
	}



	/*********************************************************************
	Safely converts String to unsigned int.

	<p><b>Details:</b> Converts a numeric string to an unsigned 32-bit integer.  Note that while the data type returned represents a signed integer, the bits of that integer reflect an unsigned conversion.</p>

	@param s the string to convert
	@param radix the radix in which the string's numeric value is expressed
	@return an Integer representing the converted value, or null if the conversion failed
	*********************************************************************/

	public final static Integer toUnsignedInt (String s, int radix)
	{
		BigInteger bi = to_int (s, radix);
		if (bi == null)
			return null;
		if
		(
			bi . compareTo (BIG_INTEGER.MAX_UINT) == 1
		||
			bi . compareTo (BIG_INTEGER.MIN_UINT) == -1
		)
			return null;
		return new Integer (bi . intValue ());
	}



}



