///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.string;



import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Stack;
import java.util.Vector;
import lava.Platform;



/******************************************************************************
Path string manipulation.

<p><b>Details:</b> PathToolbox is a collection of functions for manipulating path strings.  All paths manipulated by functions in this class are expected to be in Unix format, and all paths are returned in Unix format.</p>

@version 2000.02
@since 2000.02
@author Sharky
******************************************************************************/

public class PathToolbox
{



	/**********************************************************************
	Generates shortest equivalent path..

	<p><b>Details:</b>  <code>simplifyPath</code> modifies the given path name to represent the path's most natural form.  For example,</p>

	<blockquote>
		<code>/usr//bin/../local/./bin/</code>
	</blockquote>

	<p>is transformed into</p>

	<blockquote>
		<code>/usr/local/bin</code>&nbsp;.
	</blockquote>

	<p>Parent directory path components ("<code>..</code>") and current directory path components ("<code>.</code>") are eliminated wherever possible, and trailing slashes and redundant slashes are always removed (except where the path is simply "<code>/</code>").  Also, the parent of the root directory is assumed to be itself.  If the empty string is supplied, the returned path is "."</p>

	@param path the unnormalized path name
	@return the normamlized path name
	**********************************************************************/

	public static String simplifyPath (String path)
	{
		String[] components = StringToolbox.splitString (path, "/");
		Stack stack = new Stack ();
		if (path . startsWith ("/"))
			stack . push ("");
		int len = components . length;
		for (int i = 0; i < len; ++ i)
		{
			String comp = components [i];
			if (comp . equals ("."))
				continue;
			if (comp . equals (".."))
			{
				if (stack . isEmpty ())
				{
					// Go above base dir:
					stack . push ("..");
					continue;
				}
				Object peek = stack . peek ();
				// Parent of root is root:
				if (peek == "")
					continue;
				if (peek == "..")
				{
					// We're still climbing:
					stack . push ("..");
					continue;
				}
				// Undo last directory descended:
				stack . pop ();
				continue;
			}
			// Descend directory:
			stack . push (comp);
		}
		StringBuffer buff = new StringBuffer ();
		len = stack . size ();
		if (len == 0)
			return ".";
		buff . append (stack . firstElement ());
		for (int i = 1; i < len; ++ i)
			buff . append ('/') . append (stack . elementAt (i));
		if (buff . length () == 0)
			return "/";
		return buff . toString ();
	}



	/**********************************************************************
	Returns parent.

	<p><b>Details:</b>  getParentPath returns the parent directory of the given path (<var>path</var>).  The root directory is its own parent.  The returned path is simplified according to the rules described in <code>simplifyPath</code>.</p>

	@param path the path
	@return the path's parent
	**********************************************************************/

	public static String getParentPath (String path)
	{
		return simplifyPath (path + "/..");
	}



	/**********************************************************************
	Returns last path component.

	<p><b>Details:</b>  getFilename returns the last component of the given path (<var>path</var>).  If the empty string is supplied, the empty string is returned.</p>

	<p><b>Examples:</b></p>

	<table>
		<tr><th>input</th>           <th>output</th></tr>
		<tr><td>"/usr/local/bin"</td><td>"bin"</td> </tr>
		<td><td>"/usr/local/."</td>  <td>"."</td>   </tr>
		<td><td>"../.."</td>         <td>".."</td>  </tr>
	</table>

	@param path the path
	@return the last component
	**********************************************************************/

	public static String getFilename (String path)
	{
		if (path . length () == 0)
			return path;
		path = StringToolbox.trim (path, '/');
		if (path . equals (""))
			return "/";
		int lastslash = path . lastIndexOf ('/');
		if (lastslash < 0)
			return path;
		return path . substring (lastslash + 1);
	}



	/**********************************************************************
	Determines whether path is absolute.

	<p><b>Details:</b>  <code>isAbsolutePath</code> determines whether the given path (<var>path</var>) is an absolute path (in other words, whether the first character is '<tt>/</tt>').</p>

	@param path the path
	@return true iff the path is absolute
	**********************************************************************/

	public static boolean isAbsolutePath (String path)
	{
		return path . startsWith ("/");
	}



	/**********************************************************************
	Determines whether path is relative.

	<p><b>Details:</b>  <code>isRelativePath</code> determines whether the given path (<var>path</var>) is an relative path (in other words, whether the first character is not '<tt>/</tt>').</p>

	@param path the path
	@return true iff the path is relative
	**********************************************************************/

	public static boolean isRelativePath (String path)
	{
		return ! isAbsolutePath (path);
	}



	/**********************************************************************
	Concatenates paths.

	<p><b>Details:</b>  <code>concatenatePaths</code> concatenates the two path strings together, inserting a slash inbetween them if necessary.</p>

	@param path1 the first path string
	@param path2 the second path string
	@return the concatenated result
	**********************************************************************/

	public static String concatenatePaths (String path1, String path2)
	{
		String concatenated = path1 + '/' + path2;
		String[] parts = StringToolbox.splitString (concatenated, "/");
		StringBuffer buff = new StringBuffer (concatenated . length ());
		if (isAbsolutePath (path1))
			buff . append ('/');
		for (int i = 0; i < parts . length; ++ i)
		{
			if (i > 0)
				buff . append ('/');
			buff . append (parts [i]);
		}
		return buff . toString ();
	}



	/**********************************************************************
	Resolves relative path from working directory.

	<p><b>Details:</b>  <code></code> resolves the given relative path (<var>rel</var>) in the context of the given working directory (<var>wd</var>).  Specifically, <code>resolveRelativePath</code> returns <var>rel</var> if <var>rel</var> is an absolute path, or the concatenation (by <code>concatenatePaths</code>) of <var>wd</var> and <var>rel</var> otherwise.</p>

	@param wd the working directory
	@param rel the relative path
	@return the resolve path
	**********************************************************************/

	public static String resolveRelativePath (String wd, String rel)
	{
		if (isAbsolutePath (rel))
			return rel;
		return concatenatePaths (wd, rel);
	}



	/**********************************************************************
	Computes relative path leading from one directory to another.

	<p><b>Details:</b>  <code>toRelativePath</code> finds the shortest relative path leading from one directory (<var>from</var>) to another (<var>to</var>).  If <var>to</var> is already a relative path, then <var>to</var> is simplified and returned.  Otherwise, the relative path from <var>from</var> to <var>to</var> is computed and returned.  <var>from</var> is treated as a directory and must be absolute.</p>

	@param from the starting directory
	@param to the target directory
	@return the relative path
	**********************************************************************/

	public static String toRelativePath (String from, String to)
	{
		from = simplifyPath (from);
		to = simplifyPath (to);
		if (isRelativePath (to))
			return to;
		if (isRelativePath (from))
			throw new IllegalArgumentException ("from is relative: " + from);
		String[] fromparts = StringToolbox.splitString (from, "/");
		String[] toparts = StringToolbox.splitString (to, "/");
		int incommon = 0;
		while (incommon < fromparts . length && incommon < toparts . length)
		{
			if (! fromparts [incommon] . equals (toparts [incommon]))
				break;
			++ incommon;
		}
		Vector relparts = new Vector ();
		int level = fromparts . length;
		while (level > incommon)
		{
			relparts . addElement ("..");
			-- level;
		}
		while (level < toparts . length)
		{
			relparts . addElement (toparts [level]);
			++ level;
		}
		return toRelativePath (relparts);
	}



	private static String toRelativePath (Vector v)
	{
		if (v . isEmpty ())
			return ".";
		StringBuffer buff = new StringBuffer ();
		Enumeration enum = v . elements ();
		boolean did_one = false;
		while (enum . hasMoreElements ())
		{
			if (did_one)
				buff . append ('/');
			else
				did_one = true;
			buff . append (enum . nextElement ());
		}
		return buff . toString ();
	}



}



