///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.string;



import lava.text.html.HtmlEntities;



/******************************************************************************
Decodes strings.

<p><b>Details:</b> StringEncoder contains functions for transforming encoded strings into regular strings.</p>

@author Sharky
@version 2000.12.21
@since 2000.05.31
******************************************************************************/

public class StringDecoder
{



	/**********************************************************************
	<p><b>Details:</b> decodeCString decodes a C-style string that may contain backslash-<wbr>escaped sequences.  It is assumed that the surrounding quotes have already been stripped off.  Any unescaped illegal characters (such as quotes) will be included verbatim in the decoded string.</p>

	<p>The following characters have special interpretations when preceded by a backslash:</p>

	<blockquote>
		'a', 'b', 'f', 'n', 'r', 't', 'v', 'x'
	</blockquote>

	<p>Other characters preceded by a backslash are taken to be themselves.  (Hence, "\?" converts to just "?".)  Consult your favorite C text for more information on C string literals.</p>

	@param s the C-style string to decode
	@return the decoded string

	@version 2000.11.12
	**********************************************************************/

	public final static String decodeCString (String s)
	{
		final int
			NORMAL      = 0,
			BACKSLASH   = 1,
			OCTAL       = 2,
			HEXIDECIMAL = 3,
			UNICODE     = 4;
		int mode = NORMAL;
		char ascii_value = 0;
		int ascii_digits = 0;
		final char[] buff = s . toCharArray ();
		final int len = buff . length;
		StringBuffer decoded = new StringBuffer (len);
		for (int i = 0; i < len; ++ i)
		{
			char c = buff [i];
	restart:
			switch (mode)
			{
			case NORMAL:
				if (c == '\\')
				{
					mode = BACKSLASH;
					continue;
				}
				decoded . append (c);
				continue;

			case BACKSLASH:
				switch (c)
				{
				case 'a':
					// c = '\a';
					c = 7;
					break;
				case 'b':
					c = '\b';
					break;
				case 'f':
					c = '\f';
					break;
				case 'n':
					c = '\n';
					break;
				case 'r':
					c = '\r';
					break;
				case 't':
					c = '\t';
					break;
				case 'v':
					// c = '\v';
					c = 11;
					break;
				case 'x':
					mode = HEXIDECIMAL;
					ascii_value = 0;
					ascii_digits = 0;
					continue;
				case 'u':
					mode = UNICODE;
					ascii_value = 0;
					ascii_digits = 0;
					continue;
				case '\'':
				case '"':
				case '?':
				case '\\':
					break;
				default:
					if (c < '0' || '7' < c)
					{
						decoded . append ('\\');
						break;
					}
					mode = OCTAL;
					ascii_value = 0;
					ascii_digits = 0;
					-- i;
					continue;
				}
				decoded . append (c);
				mode = NORMAL;
				continue;

			case OCTAL:
				if (c < '0' || '7' < c)
				{
					decoded . append (ascii_value);
					mode = NORMAL;
					-- i;
					continue;
				}
				ascii_value = (char) ((ascii_value << 3) + (c - '0'));
				if (++ ascii_digits == 3)
				{
					decoded . append (ascii_value);
					mode = NORMAL;
				}
				continue;

			case HEXIDECIMAL:
				if (! isxdigit (c))
				{
					if (ascii_digits == 0)
						decoded . append ("\\x");
					else
						decoded . append (ascii_value);
					mode = NORMAL;
					-- i;
					continue;
				}
				ascii_value = (char) ((ascii_value << 4) | Character.digit (c, 16));
				if (++ ascii_digits == 2)
				{
					decoded . append (ascii_value);
					mode = NORMAL;
				}
				continue;
			case UNICODE:
				if (! isxdigit (c))
				{
					decoded . append (s . substring (i - ascii_digits - 2, i));
					mode = NORMAL;
					-- i;
					continue;
				}
				ascii_value = (char) ((ascii_value << 4) | Character.digit (c, 16));
				if (++ ascii_digits == 4)
				{
					decoded . append (ascii_value);
					mode = NORMAL;
				}
				continue;
			}
		}
		switch (mode)
		{
		case BACKSLASH:
			decoded . append ('\\');
			break;
		case HEXIDECIMAL:
			if (ascii_digits == 0)
			{
				decoded . append ("\\x");
				break;
			}
			else
				decoded . append (ascii_value);
			break;
		case UNICODE:
			decoded . append (s . substring (len - ascii_digits - 2));
			break;
		case OCTAL:
			decoded . append (ascii_value);
			break;
		default:
			break;
		}
		return decoded . toString ();
	}



	// since 2000.01.29
	private static boolean isxdigit (char c)
	{
		return
			('0' <= c && c <= '9')
		||	('a' <= c && c <= 'f')
		||	('A' <= c && c <= 'F');
	}



	/**********************************************************************
	Decodes quoted C string.

	<p><b>Details:</b>  decodeQuotedCString decodes a quoted C string literal using the common rules of C.  If the string does not begin and end with single or double quotes, or if it begins and ends with mismatching quote types, null is returned.  Otherwise, the quotes are trimmed and what's left is passed to decodeCString.</p>

	@param s the quoted string
	@return the decoded string

	@version 2000.01.29
	@since unknown
	**********************************************************************/

	public static String decodeQuotedCString (String s)
	{
		if (s == null)
			return null;
		int len = s . length ();
		if (len < 2)
			return null;
		char first = s . charAt (0);
		char last = s . charAt (len - 1);
		if (first != last)
			return null;
		if (first != '\'' && first != '"')
			return null;
		return decodeCString (s . substring (1, len - 1));
	}



	/**********************************************************************
	Converts digit character to value.

	<p><b>Details:</b>  digitForChar returns the value that would represented by the given character if the radix allowed it.  For '0'-'9', digitForChar returns 0-9; for 'A(a)'-'Z(z)', digitForChar returns 10-35.  All other characters are illegal.</p>

	@param c the character digit
	@return the digit value

	@since 2000.02.21
	**********************************************************************/

	public static int digitForChar (char c)
	{
		if ('0' <= c && c <= '9')
			return c - '0';
		if ('a' <= c && c <= 'z')
			return c + (10 - 'a');
		if ('A' <= c && c <= 'Z')
			return c + (10 - 'A');
		throw new IllegalArgumentException ("c=(char)" + (int) c);
	}



	/**********************************************************************
	Coverts hex string to bytes.

	<p><b>Details:</b> <code>hexStringToBytes</code> converts an even-<wbr>length string of hexadecimal digits into an array of <code>byte</code>s.  This operation fails if the string is <code>null</code>, the length of the string is odd, or the string contains non-<wbr>hexadecimal digits.  <code>null</code> is returned on failure.</p>

	@param s the string
	@return the array of bytes, or null if the conversion failed
	**********************************************************************/

	public static byte[] hexStringToBytes (String s)
	{
		if (s == null)
			return null;
		int count = s . length ();
		if (count % 2 != 0)
			return null;
		count /= 2;
		byte[] data = new byte [count];
		for (int i = 0; i < count; ++ i)
		{
			int i2 = i * 2;
			try
			{
				data [i] = (byte)
				(
					digitForChar (s . charAt (i2    )) << 4
				|	digitForChar (s . charAt (i2 + 1))
				);
			}
			catch (IllegalArgumentException e)
			{
				return null;
			}
		};
		return data;
	}



	/**********************************************************************
	Decodes HTML-encoded string.

	<p><b>Details:</b>  decodeHtmlText converts an HTML-encoded string (<var>encode</var>) into a natural string by decoding all of the HTML entities contained within the string.</p>

	@param the encoded string
	@return the decoded result

	@since 2000.12.21
	**********************************************************************/

	public static String decodeHtmlText (String encoded)
	{
		return HtmlEntities.decodeHtmlText (encoded);
	}



}



