///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.string;



import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import lava.clib.Ctype;
import lava.text.html.HtmlEntities;



/******************************************************************************
Encodes strings.

<p><b>Details:</b> StringEncoder contains functions for transforming regular strings into strings that satisfy various encoding rules.</p>

@version 2000.12.21
@since 2000.05.31
@author Sharky
******************************************************************************/

public class StringEncoder
{



	/**********************************************************************
	Encodes string into Java string literal.

	<p><b>Details:</b>  This method transforms a "binary" string into a Java string literal, such as might appear in Java source code, but using only 7-bit ASCII characters.  The result is contained in quotes, and all non-printable characters or characters outside the 7-bit ASCII range are encoded using appropriate Java escape sequences.  If <code>null</code> is supplied, <code>null</code> is returned.</p>

	@param input the string to encode
	@return the encoded string

	@since 1988.10.13
	**********************************************************************/

	public static String encodeAsciiJavaString (String input)
	{
		if (input == null)
			return null;
		int length = input . length ();
		StringBuffer buff = new StringBuffer (length);
		buff . append ('"');
		for (int i = 0; i < length; ++ i)
		{
			char c = input . charAt (i);
			switch (c)
			{
			case '"':
				buff . append ("\\\"");
				break;
			case '\b':
				buff . append ("\\b");
				break;
			case '\t':
				buff . append ("\\t");
				break;
			case '\n':
				buff . append ("\\n");
				break;
			case '\f':
				buff . append ("\\f");
				break;
			case '\r':
				buff . append ("\\r");
				break;
			case '\\':
				buff . append ("\\\\");
				break;
			case ' ':
				buff . append (' ');
				break;
			default:
				if (c > 0x7f || Character.isISOControl (c))
					buff . append (unicode_escape (c));
				else
					buff . append (c);
			}
		}
		return buff . append ('"') . toString ();
	}



	private static String unicode_escape (int c)
	{
		return "\\u"
		+	charForDigit (c >> 12 & 0xf)
		+	charForDigit (c >>  8 & 0xf)
		+	charForDigit (c >>  4 & 0xf)
		+	charForDigit (c       & 0xf);
	}



	/**********************************************************************
	Encodes HTML text.

	<p><b>Details:</b>  <code>encodeHtmlText</code> prepares a string for output into an HTML document by escaping the reserved characters '&lt;', '&gt;', and '&amp;'.</p>

	@param s the input string
	@return the escaped string

	@since 1999.01.29
	@version 2000.12.21
	**********************************************************************/

	public static String encodeHtmlText (String encoded)
	{
		return HtmlEntities.encodeHtmlText (encoded);
	}



	/**********************************************************************
	Converts digit value to character.

	<p><b>Details:</b>  <code>charForDigit</code> converts the given integer (<var>n</var>) into its character equivalent.  For 0 &lt;= n &lt;= 9, this is simply the character corresponding to the number.  For 10 &lt;= n &lt; 36, this is 'A' + (n - 10).  All other values for <var>n</var> are illegal.</p>

	@param n the integer
	@return the character
	**********************************************************************/

	public static char charForDigit (int n)
	{
		if (n < 0)
			throw new IllegalArgumentException ("n=" + n);
		if (n < 10)
			return (char) ('0' + n);
		if (n < 36)
			return (char) ('A' + (n - 10));
		throw new IllegalArgumentException ("n=" + n);
	}



	/**********************************************************************
	Converts byte array to hex string.

	<p><b>Details:</b>  <code>bytesToHexString</code> converts the given byte array (<var>bytes</var>) into an upper case hexadecimal string.</p>

	@param bytes the array
	@return the string

	@since 2000.01.29
	**********************************************************************/

	public static String bytesToHexString (byte[] bytes)
	{
		int len = bytes . length;
		StringBuffer buff = new StringBuffer (len * 2);
		for (int i = 0; i < len; ++ i)
		{
			int b = bytes [i] & 0xff;
			buff . append (charForDigit (b >> 4)) . append (charForDigit (b & 0xf));
		}
		return buff . toString ();
	}



}



