///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.string;



/******************************************************************************
Indentation management.

<p><b>Details:</b>  StringIndenter contains a variety of utility functions that will make dealing with indented strings and tabs in strings easier.</p>

@since 2000.01.29
@author Sharky
******************************************************************************/

public class StringIndenter
{



	/**********************************************************************
	Standard tab size.

	<p><b>Details:</b>  STANDARD_TAB_SIZE represents the traditional tab size of 8 spaces.  It is defined here for convenience.</p>
	**********************************************************************/

	public static final int STANDARD_TAB_SIZE = 8;



	/**********************************************************************
	Creates prefix for indenting.

	<p><b>Details:</b>  makeIndentPrefix creates a string of tabs and spaces suitable for prepending to a string you want to indent <var>indent</var> space.  Calling this method is similar to a call to calling <code>StringToolbox.repeat ('&nbsp;', indent)</code>, except that consecutive blocks of <var>tab_size</var> spaces are replaced with tab ('<code>\t</code>') characters.</p>

	@param indent the indent amount
	@param tab_size the tab size
	@return the indent prefix

	@since 2000.01.29
	**********************************************************************/

	public static final String makeIndentPrefix (int indent, int tab_size)
	{
		return StringToolbox.repeat ('\t', indent / tab_size) + StringToolbox.repeat (' ', indent % tab_size);
	}



	/**********************************************************************
	Creates prefix for indenting.

	<p><b>Details:</b>  makeIndentPrefix calls makeIndentPrefix (indent, STANDARD_TAB_SIZE) and returns the result.  See makeIndentPrefix(int,int) for more details.</p>

	@param indent the indent amount
	@return the indented string

	@version 1998.11.05
	**********************************************************************/

	public static final String makeIndentPrefix (int indent)
	{
		return makeIndentPrefix (indent, STANDARD_TAB_SIZE);
	}



	/**********************************************************************
	Determines indent amount.

	<p><b>Details:</b>  <code>getIndent</code> interprets the given string's leading spaces and tabs to determine the amount of indent.  Tabs count as 1-<var>tab_size</var> spaces, depending on where the tabs occur in the string.  This method recognizes only spaces and tabs.  Scanning stops at all other characters, and the indent measured up to that point is returned.</p>

	@param s the string
	@param ts the tab size
	@return the amount of indent

	@since 2000.01.29
	**********************************************************************/

	public static int getIndent (String s, int ts)
	{
		int len = s . length ();
		int indent = 0;
	scan:
		for (int i = 0; i < len; ++ i)
			switch (s . charAt (i))
			{
			case '\t':
				indent = indent / ts * ts + ts;
				break;
			case ' ':
				++ indent;
				break;
			default:
				break scan;
			}
		return indent;
	}



	/**********************************************************************
	Determines indent amount.

	<p><b>Details:</b>  <code>getIndent</code> calls getIndent (s, STANDARD_TAB_SIZE) and returns the result.  See that method for more details.</p>

	@param s the string
	@return the amount of indent

	@since 1998.11.05
	**********************************************************************/

	public static int getIndent (String s)
	{
		return getIndent (s, STANDARD_TAB_SIZE);
	}



	/**********************************************************************
	Sets indent of string.

	<p><b>Details:</b>  <code>setIndent</code> trims s of spaces (on both sides) and then prefixes the string with a number of spaces and tabs sufficient to indent the string by <var>i</var> spaces.  Tabs are assumed to be <code>ts</code> spaces.  The re-indented string is returned.</p>

	@param s the string
	@param i the amount of indent
	@param ts the tab size
	@return the re-indented string

	@since 1998.11.05
	**********************************************************************/

	public static String setIndent (String s, int i, int ts)
	{
		return makeIndentPrefix (i, ts) + s . trim ();
	}



	/**********************************************************************
	Sets indent of string.

	<p><b>Details:</b>  This method forwards to setIndent (s, i, STANDARD_TAB_SIZE).  See that method for more details.</p>

	@param s the string
	@param i the amount of indent
	@return the re-indented string

	@since 1998.11.05
	**********************************************************************/

	public static String setIndent (String s, int i)
	{
		return setIndent (s, i, STANDARD_TAB_SIZE);
	}



	/**********************************************************************
	Converts tabs to spaces.

	<p><b>Details:</b>  <code>convertTabs</code> replaces tabs in a string with the correct number of spaces to maintain the string's original spacing, assuming a tab size of <var>ts</var>.  Tabs occuring in the middle of a tab block are handled correctly.</p>

	@param s the string to convert
	@param ts the tab size
	@return the converted string

	@since 1999.01.29
	**********************************************************************/

	public static String convertTabs (String s, int ts)
	{
		final int len = s . length ();
		String pad = StringToolbox.repeat (' ', ts);
		StringBuffer out = new StringBuffer (len * 2);
		int pos = 0;
		for (int i = 0; i < len; ++ i)
		{
			char c = s . charAt (i);
			if (c != '\t')
			{
				out . append (c);
				++ pos;
			}
			else
			{
				out . append (pad);
				pos += ts;
				pos -= pos % ts;
				out . setLength (pos);
			}
		}
		return out . toString ();
	}



	/**********************************************************************
	Converts tabs to spaces.

	<p><b>Details:</b>  This implementation of convertTabs forwards to convertTabs (s, STANDARD_TAB_SIZE).  See convertTabs(String,int) for more details.</p>

	@param s the string to convert
	@return the converted string

	@since 2000.01.29
	**********************************************************************/

	public static final String convertTabs (String s)
	{
		return convertTabs (s, STANDARD_TAB_SIZE);
	}



}



