///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



package lava.text.html;



import java.util.Enumeration;
import java.util.Hashtable;



/******************************************************************************
Opening HTML tag.

<p><b>Details:</b>  This class represents opening HTML tags that have been parsed from an HTML source stream.  Instances of this class are returned by the <code>HtmlParser.parse</code> method.</p>

@see HtmlParser#parse()

@since 1999
@version 2000.12.22
******************************************************************************/

public class HtmlOpenTag
	extends HtmlRegularTag
{



	public HtmlOpenTag (String type)
	{
		super (type);
	}



	private final Hashtable ht = new Hashtable ();



	/**********************************************************************
	Represents undefined value.

	<p><b>Details:</b>  <code>NO_VALUE</code> indicates that a property was declared but that it was not given any value.</p>

	@version 2000.12.09
	**********************************************************************/

	public static final String NO_VALUE = new String ();



	/**********************************************************************
	Sets attribute.

	<p><b>Details:</b>  <code>setAttribute</code> sets the given attribute of this tag (<var>name</var>) to the given value (<var>value</var>).  To declare an attribute without assigning it a value, set <var>value</var> to <code>null</code> or NO_VALUE.  To remove an attribute, use <code>deleteAttribute</code>.</p>

	@param name attribute name
	@param value attribute value

	@since 2000.09
	@version 2000.12.09
	**********************************************************************/

	public void setAttribute (String name, String value)
	{
		if (value == null)
			value = NO_VALUE;
		ht . put (name . toUpperCase (), value);
	}



	/**********************************************************************
	Deletes attribute.

	<p><b>Details:</b>  <code>deleteAttribute</code> removes the named attribute (<var>name</var>) from this tag.  Note that this is entirely different from simply setting the attribute value to <code>null</code>.  (See <code>setAttribute</code>.)</p>

	@param
	@return
	@exception

	@since 2000.09
	@version 2000.12.09
	**********************************************************************/

	public void deleteAttribute (String name)
	{
		ht . remove (name . toUpperCase ());
	}



	/**********************************************************************
	Returns attribute.

	<p><b>Details:</b>  This method returns the attribute value associated with the given attribute name (<var>name</var>).  In cases where the original tag from the HTML stream declares an attribute name without a value, the value for that name is <code>NO_VALUE</code> (compare with <code>==</code>, not <code>String.equals</code>!).  If a tag declares more than one value for a name, only the last value is recorded.  If a tag does not have the named attribute, <code>null</code> is returned.</p>

	<p>For example, suppose this instance was formed by parsing the following (malformed) tag:</p>

	<blockquote><code>
		&lt;table cellpadding=10 align=right border cellpadding=20&gt;
	</code></blockquote>

	<p>Then, <code>getAttribute</code>, given the following names, will return the following values:</p>

	<table border=1>
		<tr>
			<th>name</th>
			<th>value</th>
			<th>explanation</th>
		</tr>
		<tr>
			<td>"TABLE"</td>
			<td>null</td>
			<td>TABLE is the tag type, but is not the name of an attribute in the tag</td>
		</tr>
		<tr>
			<td>"CELLPADDING"</td>
			<td>"20"</td>
			<td>While parsing, only the last value given for attribute CELLPADDING is stored in this instance.</td>
		</tr>
		<tr>
			<td>"ALIGN"</td>
			<td>"right"
			<td>Clearly this was the assigned value.</td>
		</tr>
		<tr>
			<td>"BORDER"</td>
			<td>""</td>
			<td>No value was given, so the object <code>NO_VALUE</code> is returned.</td>
		</tr>
	</table>

	@param name the attribute name
	@return the attribute value

	@version 2000.12.09
	**********************************************************************/

	public String getAttribute (String name)
	{
		return (String) ht . get (name . toUpperCase ());
	}



	/**********************************************************************
	Generates string representation.

	<p><b>Details:</b>  toString returns a string representation of this closing tag.  The string returned may not be exactly the same as the original text that was parsed to form this tag, but it should be equivalent by HTML standards.  Also, the attribute list may be reconstructed in an order that is different from order in which it appeared in the original HTML source.</p>

	@return the string representation
	**********************************************************************/

	public String toString ()
	{
		StringBuffer buff = new StringBuffer ();
		buff . append ("<" + type);
		for (Enumeration enum = ht . keys (); enum . hasMoreElements ();)
		{
			String name = (String) enum . nextElement ();
			buff . append (" " + name);
			String value = (String) ht . get (name);
			if (value != NO_VALUE)
				buff . append ("=\"" + HtmlEntities.encodeHtmlText (value) + "\"");
		}
		buff . append ('>');
		return buff . toString ();
	}



}



