package com.sharkysoft.printf;

import java.io.IOException;
import java.io.Writer;

/**
 * Main Printf For Java interface.
 *
 * <p><b>Details:</b> <code>Printf</code> is the primary interface to Printf For
 * Java.  This singleton class includes methods for outputing 
 * printf-<wbr>formatted text and data to the console, file streams, and 
 * buffers.</p>
 * 
 * <p>The formatting template for a printf request must be given as a 
 * <code>String</code> or a {@link PrintfTemplate}.  The data to format in a 
 * printf request must be given as an <code>Object</code> array or a 
 * {@link PrintfData}.</p>
 * 
 * <p>For more information on both of these data structures, see the 
 * <a href="doc-files/specification.htm">Printf for Java Specification 
 * 2.0</a>.</p>
 * 
 * <p>Any of the formating methods in this class may throw the following 
 * exceptions:</p>
 * 
 * <ul>
 *   <li>{@link PrintfTemplateException} - If the format template is invalid.  Or,
 *     if the format template includes a variable width or precision, and the
 *     value supplied is negative.</li>
 *   <li>{@link ClassCastException} - If a converstion type requires a certain
 *     type of data, but the wrong data type is offered.</li>
 *   <li>{@link NullPointerException} - If the data vector is <code>null</code>,
 *     but the format template calls for data.  Or, if a conversion type 
 *     requires non-<wbr><code>null</code> data, but the data being converted is 
 *     <code>null</code>.</li>
 * </ul>
 * 
 * <p><b>Quick tip for C programmers:</b> Programmers familiar with 
 * <code>printf</code> in C may find the following idiom translations 
 * useful.</p>
 * 
 * <table border=1>
 *   <tr>
 *     <th>C idiom</th>
 *     <th>Printf for Java idiom</th>
 *   </tr>
 *   <tr>
 *     <td><code>printf(fmt, arg1, arg2, ...)</code></td>
 *     <td><code>Printf.out(fmt, new Object[] {arg1, arg2, ...})</code></td>
 *   </tr>
 *   <tr>
 *     <td><code>sprintf(buff, fmt, arg1, arg2, ...)</code></td>
 *     <td><code>buff = Printf.format(fmt, new Object[] {arg1, arg2, ...})</code></td>
 *   </tr>
 *   <tr>
 *     <td><code>fprintf(out, fmt, arg1, arg2, ...)</code></td>
 *     <td><code>Printf.write(out, fmt, new Object[] {arg1, arg2, ...})</code></td>
 *   </tr>
 * </table>
 * 
 * @author Sharky 
 */
public final class Printf
{
	
  /**
   * Reserved.
   */
  private Printf()
  {
  }
  
  //////////////
  //          //
  //  format  //
  //          //
  //////////////
  
  /**
   * Formats data to string.
   * 
   * <p><b>Details:</b> <code>format</code> converts the given data into a 
   * string using the provided template.  Elements in the data vector may be 
   * overwritten if the character counting format specifier is used.  The 
   * template is pre-<wbr>compiled for efficiency.</p>
   * 
   * @param ipTemplate the template
   * @param ioapData the data
   * @return the formatted string
   * 
   * @see #format(String, Object[])
   * @see #format(String, PrintfData)
   */
  public static String format(final PrintfTemplate ipTemplate, final Object[] ioapData)
  {
    final PrintfState vpState = new PrintfState();
    vpState.mpOutput = new StringBuffer();
    vpState.mapArgs = ioapData;
    vpState.mnArgIndex = 0;
    // Iterate through characters and fmt specifiers:
    for (int vnI = 0; vnI < ipTemplate.mapComponents.length; ++vnI)
      ipTemplate.mapComponents[vnI].format(vpState);
    return vpState.mpOutput.toString();
  }
  
  /**
   * Formats data to string.
   * 
   * <p><b>Details:</b> <code>format</code> converts the given data into a 
   * string using the provided template.  Elements in the data vector may be 
   * overwritten if the character counting format specifier is used.</p>
   * 
   * @param isTemplate the template
   * @param ioapData the data
   * @return the formatted text
   * 
   * @see #format(PrintfTemplate, Object[])
   * @see #format(String, PrintfData)
   */
  public static String format(final String isTemplate, final Object[] ioapData)
  {
    return format(new PrintfTemplate(isTemplate), ioapData);
  }
  
  /**
   * Formats data to string.
   * 
   * <p><b>Details:</b> <code>format</code> converts the given data into a 
   * string using the provided template.  The data is given in a growable vector 
   * for convenience.</p>
   * 
   * @param isTemplate the template
   * @param ipData the data
   * @return the formatted string
   * 
   * @see #format(PrintfTemplate, Object[])
   * @see #format(String, Object[])
   */
  public static String format(final String isTemplate, final PrintfData ipData)
  {
    return format(new PrintfTemplate(isTemplate), ipData.done());
  }
  
	/**
	 * Formats template to stdout.
	 * 
	 * <p><b>Details:</b> <code>format</code> converts the given template to a 
	 * string.  Only format strings which do not require a data vector can be used 
	 * with this method.</p>
	 * 
   * @param isTemplate the template
   * @return the formatted string
	 * 
   * @see #format(PrintfTemplate, Object[])
   * @see #format(String, Object[])
   * @see #format(String, PrintfData)
	 */
	public static String format(final String isTemplate)
	{
		return format(isTemplate, (Object[]) null);
	}
  
  ///////////
  //       //
  //  out  //
  //       //
  ///////////
  
	/**
	 * Formats data to stdout.
	 * 
	 * <p><b>Details:</b> <code>format</code> converts the given data into a 
	 * string using the provided template and then outputs it to 
	 * <code>System.out</code>, which is usually the console.  The number of 
	 * characters output is returned.  Elements in the data vector may be 
   * overwritten if the character counting format specifier is used.  The 
   * template is pre-<wbr>compiled for efficiency.</p>
	 * 
	 * @param ipTemplate the template
	 * @param ioapData the data
	 * @return the number of characters output
	 * 
   * @see #out(String, Object[])
   * @see #out(String, PrintfData)
	 * @see #out(String)
	 */
  public static int out(final PrintfTemplate ipTemplate, final Object[] ioapData)
  {
    final String vsOut = format(ipTemplate, ioapData);
    System.out.print(vsOut);
    return vsOut.length();
  }
  
	/**
	 * Formats data to stdout.
	 * 
	 * <p><b>Details:</b> <code>format</code> converts the given data into a 
	 * string using the provided template and then outputs it to 
	 * <code>System.out</code>, which is usually the console.  The number of 
	 * characters output is returned.  Elements in the data vector may be 
	 * overwritten if the character counting format specifier is used.</p>
	 * 
	 * @param isTemplate the template
	 * @param ioapData the data
	 * @return the number of characters output
	 * 
   * @see #out(PrintfTemplate, Object[])
   * @see #out(String, PrintfData)
	 * @see #out(String)
	 */
  public static int out(final String isTemplate, final Object[] ioapData)
  {
    return out(new PrintfTemplate(isTemplate), ioapData);
  }
  
	/**
	 * Formats data to stdout.
	 * 
	 * <p><b>Details:</b> <code>format</code> converts the given data into a 
	 * string using the provided template and then outputs it to 
	 * <code>System.out</code>, which is usually the console.  The number of 
	 * characters output is returned.  The data is given in a growable vector for 
	 * convenience.</p>
	 * 
	 * @param isTemplate the template
	 * @param ipData the data
	 * @return the number of characters output
	 * 
   * @see #out(PrintfTemplate, Object[])
   * @see #out(String, Object[])
	 * @see #out(String)
	 */
  public static int out(final String isTemplate, final PrintfData ipData)
  {
    return out(isTemplate, ipData.done());
  }
  
	/**
	 * Formats template to stdout.
	 * 
	 * <p><b>Details:</b> <code>format</code> converts the given template to a 
	 * string and then outputs it to <code>System.out</code>, which is usually the 
	 * console.  The number of characters output is returned.  Only format strings 
	 * which do not require a data vector can be used with this method.</p>
	 * 
	 * @param isTemplate the template
	 * @return the number of characters output
	 * 
   * @see #out(PrintfTemplate, Object[])
   * @see #out(String, Object[])
	 * @see #out(String, PrintfData)
	 */
  public static int out(final String isTemplate)
  {
  	return out(isTemplate, (Object[]) null);
  }
  
	/////////////
	//         //
  //  write  //
  //         //
  /////////////
  
	/**
	 * Formats data to stream.
	 * 
	 * <p><b>Details:</b> <code>write</code> converts the given data into a string 
	 * using the provided template and then writes it to the given stream.  The 
	 * number of characters output is returned.  Elements in the data vector may 
	 * be overwritten if the character counting format specifier is used.  The 
	 * template is pre-<wbr>compiled for efficiency.</p>
	 * 
	 * @param ipWriter the stream
	 * @param ipTemplate the template
	 * @param ioapData the data
	 * @return the number of characters output
	 * @throws IOException if writing to the stream causes an IOException
   * 
   * @see #write(Writer, String, Object[])
   * @see #write(Writer, String, PrintfData)
   * @see #write(Writer, String)
	 */
	public static int write
	( final Writer ipWriter, 
	  final PrintfTemplate ipTemplate, 
	  final Object[] ioapData
	) throws IOException
	{
		final String vsOut = format(ipTemplate, ioapData);
		ipWriter.write(vsOut);
		return vsOut.length();
	}
  
	/**
	 * Formats data to stream.
	 * 
	 * <p><b>Details:</b> <code>write</code> converts the given data into a string 
	 * using the provided template and then outputs it to the given stream.  The 
	 * number of characters output is returned.  Elements in the data vector may 
	 * be overwritten if the character counting format specifier is used.</p>
	 * 
	 * @param ipWriter the stream
	 * @param isTemplate the template
	 * @param ioapData the data
	 * @return the number of characters output
	 * @throws IOException if writing to the stream causes an IOException
   * 
   * @see #write(Writer, PrintfTemplate, Object[])
   * @see #write(Writer, String, PrintfData)
   * @see #write(Writer, String)
	 */
	public static int write
	( final Writer ipWriter, 
	  final String isTemplate, 
	  final Object[] ioapData
	) throws IOException
	{
		return write(ipWriter, new PrintfTemplate(isTemplate), ioapData);
	}
  
	/**
	 * Formats data to stream.
	 * 
	 * <p><b>Details:</b> <code>write</code> converts the given data into a string 
	 * using the provided template and then outputs it to the given stream.  The 
	 * number of characters output is returned.  The data is given in a growable 
	 * vector for convenience.</p>
	 * 
	 * @param ipWriter the stream
	 * @param isTemplate the template
	 * @param ipData the data
	 * @return the number of characters output
	 * @throws IOException if writing to the stream causes an IOException
   * 
   * @see #write(Writer, PrintfTemplate, Object[])
   * @see #write(Writer, String, Object[])
   * @see #write(Writer, String)
	 */
	public static int write
	( final Writer ipWriter, 
	  final String isTemplate, 
	  final PrintfData ipData
	) throws IOException
	{
		return write(ipWriter, isTemplate, ipData.done());
	}
  
	/**
	 * Formats template to stream.
	 * 
	 * <p><b>Details:</b> <code>write</code> converts the given template into a 
	 * string and then outputs it to the given stream.  The number of characters 
	 * output is returned.  Only format strings which do not require a data vector 
	 * can be used with this method.</p>
	 * 
	 * @param ipWriter the stream
	 * @param isTemplate the template
	 * @return the number of characters output
	 * @throws IOException if writing to the stream causes an IOException
	 * 
   * @see #write(Writer, PrintfTemplate, Object[])
   * @see #write(Writer, String, Object[])
   * @see #write(Writer, String, PrintfData)
	 */
	public static int write(final Writer ipWriter, final String isTemplate) throws IOException
	{
		return write(ipWriter, isTemplate, (Object[]) null);
	}
  
}

