package com.sharkysoft.printf;

import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.Vector;

/**
 * Precompiled formatting template.
 *
 * <p><b>Details:</b>  A <code>PrintfTemplate</code> is a preparsed 
 * representation of a Printf for Java format string.  The constructor for this 
 * class accepts a regular <code>String</code> and parses it into a 
 * representation that is ready for use in {@link Printf}'s formatting 
 * methods.</p>
 *
 * <p>All {@link Printf} calls eventually require an instance of this class in 
 * order to generate output.  The instance can either be created 
 * <em>explicitly</em> by the programmer or <em>automatically</em> by Printf for 
 * Java.  In other words, when the caller passes a format string to 
 * {@link Printf}, it can be sent as a either a <code>String</code> or a 
 * <code>PrintfTemplate</code>.  This is demonstrated in the examples 
 * below:</p>
 *
 * <blockquote class="example">
 *
 *   <p><b>Declarations:</b></p>
 *
    *<blockquote><pre>
      *String name = "Sharky";
      *int age = 27;
      *Object[] data = new Object[] {name, new Integer(age)};
    *</pre></blockquote>
 *
 *   <p><b>Example 1:</b> Sending the <code>printf</code> format string
 *   as a <code>String</code>:</p>
 *
    *<blockquote><pre>
      *Printf.out("My friend %s is %d years old.\n", data);
    *</pre></blockquote>
 *
 *   <p><b>Example 2:</b> Sending the <code>printf</code> format string
 *   as a <code>PrintfTemplate</code>:</p>
 *
    *<blockquote><pre>
      *Printf.out
      *( new PrintfTemplate("My friend %s is %d years old.\n"), 
      *  data
      *);
    *</pre></blockquote>
 *
 * </blockquote>
 *
 * <p>Examples 1 and 2 produce <em>exactly</em> the same output, and require
 * approximately the same amount of execution time and system resources.  The
 * only difference between the two is that the first example requires
 * {@link Printf} to automatically generate a <code>PrintfTemplate</code> 
 * <em>during</em> the call, whereas the second example creates one 
 * <em>prior</em> to the call.  Other factors being equal, the first example is 
 * preferred simply because it is more readable.</p>
 *
 * <p>However, if the call to {@link Printf} is inside a loop, where multiple 
 * lines of output are produced using exactly the same formatting template, then 
 * it is probably more efficient to parse the format string just once, prior to 
 * entering the loop, and then reuse the parsed result.  The only way to 
 * accomplish this is to explicitely initialize the 
 * <code>PrintfTemplate</code> outside of the loop.</p>
 *
 * <blockquote class="example">
 *
 *   <p><b>Example 3:</b> Optimizing a print loop by compiling the format string
 *   <em>outside</em> the loop:</p>
 *
    *<blockquote><pre>
      *PrintfTemplate format_string = 
      *  new PrintfTemplate("%d bottles of beer on the wall.\n");
      *for (int i = 99; i &gt; 0; -- i)
      *  Printf.out
      *  ( format_string,
      *    new Object[]
      *    { new Integer (i)
      *    }
      *  );
    *</pre></blockquote>
 *
 * </blockquote>
 *
 * <p>Preliminary measurements have shown that using precompiled format strings
 * can speed up printf loops by 10-20%, but this of course depends on the 
 * complexity of the format string.</p>
 * 
 * <p>For the complete specification governing the syntax of valid printf format 
 * strings, see 
 * <a href="doc-files/specification.htm">Printf for Java Specification 2.0</a>.</p>
 *
 * @see Printf
 * 
 * @author Sharky
 */
public final class PrintfTemplate
{

  /*
   * Format string subcomponents.
   */
  final Formatter[] mapComponents;

  /*
   * Special value for mnArgsRequired, indicating that the true value for
   * args_required has not yet been determined.
   */
  private static final int UNDETERMINED = -1;
  
  /*
   * The length of the argument list required for this format string.
   */
  int mnArgsRequired = UNDETERMINED;
  
  private final String msAsString;
  
  /**
   * Parses format string.
   * 
   * <p><b>Details:</b> This constructor parses the given format string and 
   * assembles a corresponding formatting engine.  If the format string is 
   * invalid, a {@link PrintfTemplateException} will be thrown.</p> 
   * 
   * @param isFormat the format string
   * @exception PrintfTemplateException if the format string is invalid
   */
  public PrintfTemplate(final String isFormat)
  {
    msAsString = isFormat;
    final Vector vpVector = new Vector();
    final StringCharacterIterator vpIter = new StringCharacterIterator(isFormat);
    while (vpIter.current() != CharacterIterator.DONE)
      vpVector.addElement
      ( Formatter.create
        ( new FormatSpecifier(vpIter)
        )
      );
    mapComponents = new Formatter[vpVector.size()];
    vpVector.copyInto(mapComponents);
  }
  
  /**
   * Determines data vector length.
   * 
   * <p><b>Details:</b> <code>argsRequired</code> determines and returns the 
   * minimum length of a suitable data vector.</p>
   * 
   * @return the minimum length
   */
  public int argsRequired()
  {
    if (mnArgsRequired == UNDETERMINED)
    {
      mnArgsRequired = 0;
      for (int vnI = 0; vnI < mapComponents.length; ++vnI)
        mnArgsRequired += mapComponents[vnI].argsRequired();
    }
    return mnArgsRequired;
  }
  
  /**
   * Returns original format string.
   * 
   * <p><b>Details:</b> <code>toString</code> returns the original format string
   * used to initialize this instance.</p>
   * 
   * @return the format string
   */
  public String toString()
  {
    return msAsString;
  }
  
}

