package com.sharkysoft.printf.engine;

import com.sharkysoft.util.EnumeratedType;

/**
 * Text cropping styles.
 *
 * <p><b>Details:</b> <code>CroppingMode</code> is an enumeration of text
 * cropping modes.  Cropping occurs when text larger than the field in which it
 * is displayed.</p>
 *
 * @author Sharky
 */
public final class CroppingMode extends EnumeratedType
{

  /**
   * Initializes type-safe object value.
   *
   * <p><b>Details:</b> This constructor initializes type-<wbr>safe object
   * values.</p>
   *
   * @param inValue magic int value
   * @param isString string form
   */
  private CroppingMode(int inValue, String isString)
  {
    super(inValue, isString);
  }

  /**
   * No cropping.
   *
   * <p><b>Details:</b> <code>NONE</code> indicates the text is not cropped.</p>
   */
  public static final int NONE = 0;

  /**
   * No cropping.
   *
   * <p><b>Details:</b> <code>gpNone</code> is a type-<wbr>safe representation
   * of <code>NONE</code>.</p>
   */
  public static final CroppingMode gpNone = new CroppingMode(NONE, "NONE");

  /**
   * Right cropping.
   *
   * <p><b>Details:</b> <code>RIGHT</code> indicates that the right-<wbr>most
   * portion of the text is cropped.</p>
   */
  public static final int RIGHT = 1;

  /**
   * Right cropping.
   *
   * <p><b>Details:</b> <code>gpRight</code> is a type-<wbr>safe representation
   * of <code>RIGHT</code>.</p>
   */
  public static final CroppingMode gpRight = new CroppingMode(RIGHT, "RIGHT");

  /**
   * Left cropping.
   *
   * <p><b>Details:</b> <code>LEFT</code> indicates that the left-<wbr>most
   * portion of the text is cropped.</p>
   */
  public static final int LEFT = 2;

  /**
   * Left cropping.
   *
   * <p><b>Details:</b> <code>gpLeft</code> is a type-<wbr>safe representation
   * of <code>LEFT</code>.</p>
   */
  public static final CroppingMode gpLeft = new CroppingMode(LEFT, "LEFT");

  /**
   * Middle cropping.
   *
   * <p><b>Details:</b> <code>MIDDLE</code> indicates that a center portion of
   * the text may be cropped (and perhaps replaced with ellipses).</p>
   */
  public static final int MIDDLE = 3;

  /**
   * Middle cropping.
   *
   * <p><b>Details:</b> <code>gpMiddle</code> is a type-<wbr>safe representation
   * of <code>MIDDLE</code>.</p>
   */
  public static final CroppingMode gpMiddle = new CroppingMode(MIDDLE, "MIDDLE");

  /**
   * Converts int value to object value.
   *
   * <p><b>Details:</b> <code>forInt</code> returns the type-<wbr>safe object
   * value corresponding to the given <code>int</code> value.</p>
   *
   * @param inValue int value
   * @return type-safe object value
   */
  public static CroppingMode forInt(int inValue)
  {
    return (CroppingMode) EnumeratedType.toEnumeratedType(inValue);
  }

}

