package com.sharkysoft.printf.engine;

import java.math.BigDecimal;
import java.math.BigInteger;

import com.sharkysoft.string.StringToolbox;

/**
 * Formats integers.
 *
 * <p>An <code>IntegerFormatter</code> formats integers.  See
 * {@link NumberFormatter} for a generalized discussion relating to this
 * class.</p>
 *
 * <blockquote class="example">
 *
 *   <p><b>Example:</b> Output the contents of integer array <var>vals</var>,
 *   one integer per line, right-<wbr>justified in a 25-<wbr>character-<wbr>wide
 *   field.  Round each integer to 3 significant digits.</p>
 *
    *<blockquote><pre>
      *int[] vals = {2000, 34, 32768};
      *IntegerFormatter formatter = new IntegerFormatter();
      *formatter.setFieldWidth(25);
      *formatter.setPadChar(':'); // non-space chosen for emphasis
      *formatter.setAlignment(AlignmentMode.gpRight);
      *formatter.setSigDigits(3);
      *for (int i = 0; i &lt; vals.length; ++ i)
      *  System.out.println(formatter.format(vals[i]));
    *</pre></blockquote>
 *
 *   <p>This produces the following output:</p>
 *
    *<blockquote><pre>
      *:::::::::::::::::::::2000
      *:::::::::::::::::::::::34
      *::::::::::::::::::::32800
    *</pre></blockquote>
 *
 * </blockquote>
 *
 * <p>The formatting properties for a given instance of this class can be
 * updated at any time, and doing so changes the way that subsequent data is
 * formatted.  For a detailed description of all formatting properties, review
 * the documentation below and in the superclass.</p>
 *
 * @author Sharky
 */
public class IntegerFormatter extends NumberFormatter
{

  ////////////////
  //            //
  //  PadDigit  //
  //            //
  ////////////////

  /**
   * Pad digit.
   *
   * <p>Details:</b> Property <code>PadDigit</code> is the pad digit to use when
   * extra digits are required because of the <code>MinDigits</code> property.
   * <b>Default value:</b> '0'.</p>
   *
   * @see #getPadDigit()
   * @see #setPadDigit(char)
   */
  protected char mcPadDigit = '0';

  /**
   * Retrieves PadDigit property.
   *
   * @return current value
   *
   * @see #mcPadDigit
   */
  public char getPadDigit()
  {
    return mcPadDigit;
  }

  /**
   * Updates PadDigit property.
   *
   * @param icPadDigit new value
   *
   * @see #mcPadDigit
   */
  public void setPadDigit(final char icPadDigit)
  {
    mcPadDigit = icPadDigit;
  }

  /////////////////
  //             //
  //  MinDigits  //
  //             //
  /////////////////

  /**
   * Minimum digits.
   *
   * <p><b>Details:</b> Property <code>MinDigits</code> is the minimum number of
   * digits to display in the formatted output.  If the number isn't large
   * enough to fill all the digits, the padding digit will be used to fill out
   * the rest.  If this property is 0, digits will only be displayed if the
   * value being rendered is non-zero.  <b>Default value:</b> 1.</p>
   *
   * @see #getMinDigits()
   * @see #setMinDigits(int)
   */
  protected int mnMinDigits  = 1;

  /**
   * Retrieves MinDigits property.
   *
   * @return current value
   *
   * @see #mnMinDigits
   */
  public int getMinDigits()
  {
    return mnMinDigits;
  }

  /**
   * Updates MinDigits property.
   *
   * @param inMinDigits new value
   *
   * @see #mnMinDigits
   */
  public void setMinDigits(final int inMinDigits)
  {
    if (inMinDigits < 0)
      throw new IllegalArgumentException("inMinDigits=" + inMinDigits);
    mnMinDigits = inMinDigits;
  }

  /**
   * Default constructor.
   */
  public IntegerFormatter() {}

  /**
   * Renders value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given integer value into
   * a numeric string, formatted according to the criteria set forth in the
   * properties.</p>
   *
   * @param ipValue value to render
   * @return rendered value
   */
  public String format(BigInteger ipValue)
  {
    if (mnSigDigits > 0)
      ipValue = setSignificantDigits(new BigDecimal(ipValue), mnSigDigits).toBigInteger();
    final String vsPrefix;
    final String vsSuffix;
    String vsDigits;
    switch (ipValue.signum())
    {
    case +1:
      vsPrefix = msPosPrefix;
      vsSuffix = msPosSuffix;
      vsDigits = ipValue.toString(mnRadix);
      break;
    case 0:
      vsPrefix = msZeroPrefix;
      vsSuffix = msZeroSuffix;
      vsDigits = mnMinDigits > 0 ? "0" : "";
      break;
    case -1:
      vsPrefix = msNegPrefix;
      vsSuffix = msNegSuffix;
      vsDigits = ipValue.negate().toString(mnRadix);
      break;
    default:
      throw new RuntimeException ("unreachable code");
    }
    if (mzUpperCase)
      vsDigits = vsDigits.toUpperCase();
    // Make extender:
    final String vsExtender;
    final int vsExtra = mnMinDigits - vsDigits.length();
    if (vsExtra > 0)
    {
      if (mcPadDigit != 0)
        vsExtender = StringToolbox.repeat(mcPadDigit, vsExtra);
      else
        vsExtender = StringToolbox.repeat(' ', vsExtra);
    }
    else
      vsExtender = "";
    StringBuffer vsLeftSide = new StringBuffer();
    StringBuffer vsRightSide = new StringBuffer();
    // Build leader:
    if (mnAlignment == AlignmentMode.FULL)
    {
      vsLeftSide.append(vsPrefix);
      vsRightSide.append(vsExtender);
    }
    else
    {
      if (mcPadDigit != 0)
      {
        vsRightSide.append(vsPrefix);
        vsRightSide.append(vsExtender);
      }
      else
      {
        vsRightSide.append(vsExtender);
        vsRightSide.append(vsPrefix);
      }
    }
    // Append actual value:
    vsRightSide.append(vsDigits);
    // Concatenate the two sides together and finish formatting:
    return format(vsLeftSide.toString(), vsRightSide.toString());
  }

  /**
   * Renders value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given integer value into
   * a numeric string, formatted according to the criteria set forth in the
   * properties.</p>
   *
   * @param ilValue value to render
   * @return rendered value
   */
  public final String format(final long ilValue)
  {
    return format(BigInteger.valueOf(ilValue));
  }

  /**
   * Renders value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given integer value into
   * a numeric string, formatted according to the criteria set forth in the
   * properties.</p>
   *
   * @param inValue value to render
   * @return rendered value
   */
  public final String format(final int inValue)
  {
    return format(BigInteger.valueOf(inValue));
  }

  //////////////////////////
  //                      //
  //  unsigned rendering  //
  //                      //
  //////////////////////////

  /**
   * Long MSB manipulation constant.
   *
   * <p><b>Details:</b> This constant is used by <code>formatUnsigned</code> to
   * manipulate the sign bit of a long value into the value's most significant
   * bit.  Refer to the source for <code>formatUnsigned(long)</code> for more
   * details.</p>
   */
  private static final BigInteger LONG_NEG_ADJUSTMENT = new BigInteger("ffffffffffffffff", 16).not();

  /**
   * Renders unsigned value.
   *
   * <p><b>Details:</b> <code>formatUnsigned</code> renders the given integer
   * value as a numeric string, formatted according to the criteria set forth in
   * the properties.  The integer value is interpreted as an unsigned value.</p>
   *
   * @param ilValue value to render
   * @return rendered value
   */
  public final String formatUnsigned(final long ilValue)
  {
    BigInteger vpBig = BigInteger.valueOf(ilValue);
    if (ilValue < 0)
      vpBig = vpBig.xor(LONG_NEG_ADJUSTMENT);
    return format(vpBig);
  }

  /**
   * Renders unsigned value.
   *
   * <p><b>Details:</b> <code>formatUnsigned</code> renders the given integer
   * value as a numeric string, formatted according to the criteria set forth in
   * the properties.  The integer value is interpreted as an unsigned value.</p>
   *
   * @param inValue value to render
   * @return rendered value
   */
  public final String formatUnsigned(final int inValue)
  {
    return format(BigInteger.valueOf(inValue & 0xffffffffL));
  }

  /**
   * Renders unsigned value.
   *
   * <p><b>Details:</b> <code>formatUnsigned</code> renders the given integer
   * value as a numeric string, formatted according to the criteria set forth in
   * the properties.  The integer value is interpreted as an unsigned value.</p>
   *
   * @param iwValue value to render
   * @return rendered value
   */
  public final String formatUnsigned(final short iwValue)
  {
    return format(BigInteger.valueOf(iwValue & 0xffffL));
  }

  /**
   * Renders unsigned value.
   *
   * <p><b>Details:</b> <code>formatUnsigned</code> renders the given integer
   * value as a numeric string, formatted according to the criteria set forth in
   * the properties.  The integer value is interpreted as an unsigned value.</p>
   *
   * @param ibValue value to render
   * @return rendered value
   */
  public final String formatUnsigned(byte ibValue)
  {
    return format(BigInteger.valueOf(ibValue & 0xffL));
  }

}

