package com.sharkysoft.printf;

import java.text.CharacterIterator;
import java.text.StringCharacterIterator;

final class FormatSpecifier
{
	static final char 
		JUSTIFICATION_RIGHT = 0, 
		JUSTIFICATION_LEFT = '-', 
		JUSTIFICATION_CENTER = '^', 
		JUSTIFICATION_FULL = '0', 
		JUSTIFICATION_UNSPECIFIED = JUSTIFICATION_RIGHT;

	char mcJustification = JUSTIFICATION_UNSPECIFIED;

	static final char 
		PREFIX_NONE = 0, 
		PREFIX_PLUS = '+', 
		PREFIX_SPACE = ' ', 
		PREFIX_UNSPECIFIED = PREFIX_NONE;
		
	char mcPositivePrefix = PREFIX_UNSPECIFIED;
	
	static final char 
		ALTERNATE_OFF = 0, 
		ALTERNATE_ON = '#', 
		ALTERNATE_UNSPECIFIED = ALTERNATE_OFF;
		
	char mcAlternate = ALTERNATE_UNSPECIFIED;
	
	static final char 
		PADCHAR_SPACE = ' ', 
		PADCHAR_ZERO = '0', 
		PADCHAR_UNSPECIFIED = PADCHAR_SPACE;
		
	char mcPadChar = PADCHAR_UNSPECIFIED;

	static final int 
		WIDTH_DEFAULT = -2, 
		WIDTH_ARGUMENT = -1, 
		WIDTH_UNSPECIFIED = WIDTH_DEFAULT;
		
	int mnWidth = WIDTH_UNSPECIFIED;
	
	static final int 
		PRECISION_ANY = -2, 
		PRECISION_ARGUMENT = -1, 
		PRECISION_ZERO = 0, 
		PRECISION_UNSPECIFIED = PRECISION_ANY;
		
	int mnPrecision = PRECISION_UNSPECIFIED;
	
	static final char 
		INPUT_SIZE_NORMAL = 0, 
		INPUT_SIZE_SHORT = 'h', 
		INPUT_SIZE_LONG = 'l', 
		// INPUT_SIZE_LONG_DOUBLE = 'L',
		INPUT_SIZE_BYTE = 'b', 
		INPUT_SIZE_BIG = 'B', 
		INPUT_SIZE_UNSPECIFIED = INPUT_SIZE_NORMAL;

	char mcInputSize = INPUT_SIZE_UNSPECIFIED;

	static final char 
		TYPE_SIGNED_DEC = 'd', 
		TYPE_UNSIGNED_DEC = 'u', 
		TYPE_OCT = 'o', 
		TYPE_HEX_LOWER = 'x', 
		TYPE_HEX_UPPER = 'X', 
		TYPE_FLOAT = 'f', 
		TYPE_FLOAT_E_LOWER = 'e', 
		TYPE_FLOAT_E_UPPER = 'E', 
		TYPE_FLOAT_G_LOWER = 'g', 
		TYPE_FLOAT_G_UPPER = 'G', 
		TYPE_CHAR = 'c', 
		TYPE_STRING = 's', 
		TYPE_PERCENT = '%', 
		TYPE_ENDL = '\n', 
		TYPE_CHARCOUNT = 'n', 
		TYPE_POINTER = 'p', 
		TYPE_INTEGER_UPPER = 'Z', 
		TYPE_INTEGER_LOWER = 'z', 
		TYPE_UNSPECIFIED = 0, 
		TYPE_LITERAL = 0;

	char mcType = TYPE_UNSPECIFIED;

	int mnBase = 0;

	final String msLiteral;

	String msError = null;

	private void setError(final String isError)
	{
		if (msError == null)
			msError = isError;
	}

	FormatSpecifier(final StringCharacterIterator ipFs)
	{
		final PrintfStringCharacterIterator vpFs = new PrintfStringCharacterIterator(ipFs);
		char vcC;
		vcC = vpFs.current();
		if (vcC != '%')
		{
			final StringBuffer vpBuff = new StringBuffer();
			while (vcC != CharacterIterator.DONE && vcC != '%')
			{
				vpBuff.append(vcC);
				vcC = vpFs.next();
			}
			msLiteral = vpBuff.toString();
			return;
		}
		// Parse various fields of the format spec.  At the entrance to each parsing 
		// subsection, c contains the next character to be parsed.
		vcC = vpFs.next();
		// Parse flag characters (optional):
		parse_flags : while (true)
		{
			switch (vcC)
			{
			case JUSTIFICATION_LEFT:
				if (mcJustification != JUSTIFICATION_UNSPECIFIED)
					setError("multiple justifications");
				mcJustification = JUSTIFICATION_LEFT;
				break;
			case JUSTIFICATION_CENTER:
				if (mcJustification != JUSTIFICATION_UNSPECIFIED)
					setError("multiple justifications");
				mcJustification = JUSTIFICATION_CENTER;
				break;
			case PADCHAR_ZERO:
				if (mcJustification != JUSTIFICATION_UNSPECIFIED)
					setError("multiple justifications");
				mcPadChar = PADCHAR_ZERO;
				mcJustification = JUSTIFICATION_FULL;
				break;
			case PREFIX_PLUS:
			case PREFIX_SPACE:
				if (mcPositivePrefix != PREFIX_UNSPECIFIED)
					setError("multiple positive prefixes");
				mcPositivePrefix = vcC;
				break;
			case ALTERNATE_ON:
				if (mcAlternate != ALTERNATE_UNSPECIFIED)
					setError("alternate format selected twice");
				mcAlternate = ALTERNATE_ON;
				break;
			default:
				break parse_flags;
			}
			vcC = vpFs.next();
		}
		// Parse width spec (optional):
		switch (vcC)
		{
		case '*':
			mnWidth = WIDTH_ARGUMENT;
			vcC = vpFs.next();
			break;
		default:
			if (!isdigit(vcC))
			{
				if (mcPadChar != PADCHAR_UNSPECIFIED)
					setError("padding specified without field width");
				if (mcJustification != JUSTIFICATION_UNSPECIFIED)
					setError("justification specified without field width");
				break;
			}
			int vnN = 0;
			do
			{
				vnN = vnN * 10 + (vcC - '0');
				vcC = vpFs.next();
			}
			while (isdigit(vcC));
			mnWidth = vnN;
		}
		// Parse precision spec (optional):
		if (vcC == '.')
		{
			vcC = vpFs.next();
			if (vcC == PRECISION_ZERO)
			{
				mnPrecision = PRECISION_ZERO;
				vcC = vpFs.next();
			}
			else if (isdigit(vcC))
			{
				int n = 0;
				do
				{
					n = n * 10 + (vcC - '0');
					vcC = vpFs.next();
				}
				while (isdigit(vcC));
				mnPrecision = n;
			}
			else if (vcC == '*')
			{
				mnPrecision = PRECISION_ARGUMENT;
				vcC = vpFs.next();
			}
			else
				setError("invalid precision specifier");
		}
		// Parse input size modifier (optional):
		switch (vcC)
		{
		case INPUT_SIZE_LONG:
		case INPUT_SIZE_SHORT:
			// case INPUT_SIZE_LONG_DOUBLE:
		case INPUT_SIZE_BYTE:
		case INPUT_SIZE_BIG:
			mcInputSize = vcC;
			vcC = vpFs.next();
		}
		// Parse conversion-type character (required):
		mcType = vcC;
		if (mcType == 'i')
			mcType = 'd';
		switch (mcType)
		{
		case TYPE_UNSIGNED_DEC :
			disallowPositivePrefix();
			disallowAlternateFormat();
			break;
		case TYPE_SIGNED_DEC :
			disallowAlternateFormat();
			break;
		case TYPE_CHAR :
			disallowPrecision();
			disallowPositivePrefix();
			disallowFullJustification();
			break;
		case TYPE_STRING :
			disallowAlternateFormat();
			disallowPositivePrefix();
			disallowFullJustification();
			break;
		case TYPE_POINTER :
			disallowAlternateFormat();
			disallowPositivePrefix();
			break;
		case TYPE_INTEGER_UPPER :
		case TYPE_INTEGER_LOWER :
			disallowPositivePrefix();
			disallowAlternateFormat();
			if (vpFs.next() != '[')
			{
				setError("'" + mcType + "' not followed by '['");
				break;
			}
			StringBuffer digits = new StringBuffer();
			while (true)
			{
				vcC = vpFs.next();
				if (vcC == ']')
					break;
				if (vcC < '0' || '9' < vcC)
					throw new PrintfTemplateException("illegal base characters for '" + mcType + "[]'");
				digits.append(vcC);
			}
			if (digits.length() == 0)
				throw new PrintfTemplateException("empty base for '" + mcType + "[]'");
			mnBase = Integer.parseInt(digits.toString());
			if (mnBase < 2 || mnBase > 36)
				throw new PrintfTemplateException("illegal base for " + mcType + "[]: " + mnBase);
		case TYPE_HEX_LOWER :
		case TYPE_HEX_UPPER :
			disallowPositivePrefix();
			break;
		case TYPE_OCT :
			disallowPositivePrefix();
			break;
		case TYPE_FLOAT :
		case TYPE_FLOAT_E_LOWER :
		case TYPE_FLOAT_E_UPPER :
		case TYPE_FLOAT_G_LOWER :
		case TYPE_FLOAT_G_UPPER :
			disallow_short_size();
			break;
		case TYPE_PERCENT :
		case TYPE_ENDL :
		case TYPE_CHARCOUNT :
			disallow_size();
			disallowAlternateFormat();
			disallowPositivePrefix();
			disallowWidth();
			disallowPrecision();
			break;
		default :
			setError("conversion type character missing or invalid");
		}
		// Set the iterator to the start of the next substring:
		vpFs.next();
		msLiteral = vpFs.getSubstring();
		if (msError != null)
			throw new PrintfTemplateException("invalid format specifier: \"" + msLiteral + "\" (" + msError + ")");
	}

	private void disallow_size()
	{
		if (mcInputSize != INPUT_SIZE_NORMAL)
			setError("input size not allowed for %" + mcType);
	}

	private void disallow_short_size()
	{
		switch (mcInputSize)
		{
			case INPUT_SIZE_SHORT :
			case INPUT_SIZE_BYTE :
				setError("short or byte input size not allowed for %" + mcType);
		}
	}

	private void disallowAlternateFormat()
	{
		if (mcAlternate != ALTERNATE_UNSPECIFIED)
			setError("alternate format not allowed for %" + mcType);
	}

	private void disallowPositivePrefix()
	{
		if (mcPositivePrefix != PREFIX_UNSPECIFIED)
			setError("positive prefix not allowed for %" + mcType);
	}

	private void disallowFullJustification()
	{
		if (mcJustification == JUSTIFICATION_FULL)
			setError("full justification not allowed for %" + mcType);
	}

	private void disallowWidth()
	{
		if (mnWidth != WIDTH_UNSPECIFIED)
			setError("field width not allowed for %" + mcType);
	}

	private void disallowPrecision()
	{
		if (mnPrecision != PRECISION_UNSPECIFIED)
			setError("mnPrecision not allowed for %" + mcType);
	}

	private void disallowType()
	{
		setError("sorry, %" + mcType + " not supported");
	}

	private String toNormalizedString()
	{
		final StringBuffer s = new StringBuffer("%");
		if (mcJustification != JUSTIFICATION_UNSPECIFIED)
			s.append(mcJustification);
		if (mcPadChar != PADCHAR_UNSPECIFIED)
			s.append(mcPadChar);
		if (mcAlternate != ALTERNATE_UNSPECIFIED)
			s.append(mcAlternate);
		if (mnWidth != WIDTH_UNSPECIFIED)
		{
			if (mnWidth == WIDTH_ARGUMENT)
				s.append('*');
			else
				s.append(mnWidth);
		}
		if (mnPrecision != PRECISION_UNSPECIFIED)
		{
			s.append('.');
			if (mnPrecision == PRECISION_ARGUMENT)
				s.append('*');
			else
				s.append(mnPrecision);
		}
		if (mcInputSize != INPUT_SIZE_UNSPECIFIED)
			s.append(mcInputSize);
		switch (mcType)
		{
			case '\n' :
				s.append("\\n");
				break;
			default :
				s.append(mcType);
		}
		switch (mcType)
		{
			case TYPE_INTEGER_UPPER :
			case TYPE_INTEGER_LOWER :
				s.append("[" + mnBase + "]");
		}
		return s.toString();
	}

	public String toString()
	{
		return msLiteral;
	}

	/**
	 * <p><b>Details:</b> Returns <code>true</code> if inC is a decimal digit.  
	 * Returns the result of <code>Character.isDigit(c)</code>.</p>
	 */
	static boolean isdigit(final int inC)
	{
		return Character.isDigit((char) inC);
	}
}

