package com.sharkysoft.printf.engine;

import com.sharkysoft.util.EnumeratedType;

/**
 * Text justification styles.
 *
 * <p><b>Details:</b> <code>AlignmentMode</code> is an enumeration of alignment
 * modes supported by {@link StringFormatter}.  When the text is shorter than
 * the field into which it is being rendered, it may be justified against the 
 * left edge of the field, the right edge of the field, or both.  The text may 
 * also be centered.  This class represents each of the alignment modes with 
 * both <code>int</code>s and type-<wbr>safe object values.</p>
 *
 * @see StringFormatter
 * @author Sharky
 */
public final class AlignmentMode extends EnumeratedType
{

  /**
   * Initializes type-safe object value.
   *
   * <p><b>Details:</b> This constructor initializes type-safe object
   * values.</p>
   *
   * @param inValue magic int value
   * @param isString string form
   */
  private AlignmentMode(int inValue, String isString)
  {
    super(inValue, isString);
  }

  /**
   * Left-alignment.
   *
   * <p><b>Details:</b> <code>LEFT</code> indicates that text is
   * left-<wbr>aligned in the field.</p>
   *
   * @see #gpLeft
   */
  public static final int LEFT = 1;

  /**
   * Left-alignment.
   *
   * <p><b>Details:</b> <code>gpLeft</code> is a type-<wbr>safe representation
   * of <code>LEFT</code>.</p>
   *
   * @see #LEFT
   */
  public static final AlignmentMode gpLeft = new AlignmentMode(LEFT, "LEFT");

  /**
   * Centered alignment.
   *
   * <p><b>Details:</b> <code>CENTER</code> indicates that text is centered in
   * the field.</p>
   *
   * @see #gpCenter
   */
  public static final int CENTER = 2;

  /**
   * Centered alignment.
   *
   * <p><b>Details:</b> <code>gpCenter</code> is a type-<wbr>safe representation
   * of <code>CENTER</code>.</p>
   *
   * @see #CENTER
   */
  public static final AlignmentMode gpCenter = new AlignmentMode(CENTER, "CENTER");

  /**
   * Right-alignment.
   *
   * <p><b>Details:</b> <code>RIGHT</code> indicates that text is
   * right-<wbr>aligned in the field.</p>
   *
   * @see #gpRight
   */
  public static final int RIGHT = 3;

  /**
   * Right-alignment.
   *
   * <p><b>Details:</b> <code>gpRight</code> is a type-<wbr>safe representation
   * of <code>RIGHT</code>.</p>
   *
   * @see #RIGHT
   */
  public static final AlignmentMode gpRight = new AlignmentMode(RIGHT, "RIGHT");

  /**
   * Full justification.
   *
   * <p><b>Details:</b> <code>FULL</code> indicates that text is fully justified
   * in the field.  Fully justified text is aligned to both the left and right
   * sides of the field.  It may be necessary to insert spaces into the text in
   * order to make it the same width as the field.</p>
   *
   * @see #gpFull
   */
  public static final int FULL = 4;

  /**
   * Full justification.
   *
   * <p><b>Details:</b> <code>gpFull</code> is a type-<wbr>safe representation
   * of <code>FULL</code>.</p>
   *
   * @see #FULL
   */
  public static final AlignmentMode gpFull = new AlignmentMode(FULL, "FULL");

  /**
   * Converts int value to object value.
   *
   * <p><b>Details:</b> <code>forInt</code> returns the type-<wbr>safe object
   * value corresponding to the given <code>int</code> value.</p>
   *
   * @param inValue int value
   * @return type-safe object value
   */
  public static AlignmentMode forInt(int inValue)
  {
    return (AlignmentMode) EnumeratedType.toEnumeratedType(inValue);
  }

}

