package com.sharkysoft.printf.engine;

import java.math.BigDecimal;

/**
 * Formats numbers.
 *
 * <p>A <code>NumberFormatter</code> formats numbers in a field by adjusting
 * their precision and padding them as necessary, according to criteria selected
 * by the client.  Properties control various formatting constraints, such as
 * field width, justification style, padding characters, and so on.</p>
 *
 * <p>Because applications that format numeric values frequently format multiple
 * values with the same criteria, this class is designed to "remember" selected
 * formatting properties so that the settings they can be reused with each
 * subsequent formatting request.  This approach minimizes the number of
 * parameters that must be passed with each formatting call and results in more
 * efficient execution for large formatting jobs.</p>
 *
 * <p><code>NumberFormatter</code> extends {@link StringFormatter} with abstract
 * number formatting constraints that are common to {@link IntegerFormatter} and
 * {@link RealFormatter}.  This class does not actually implement any of the
 * specific numeric formatting rules, but leaves that work to the
 * subclasses.</p>
 *
 * @see IntegerFormatter
 * @see RealFormatter
 *
 * @author Sharky
 */
public abstract class NumberFormatter extends StringFormatter
{

  /////////////
  //         //
  //  Radix  //
  //         //
  /////////////

  /**
   * Radix.
   *
   * <p><b>Details:</b> Property <code>Radix</code> is the output radix (numeric
   * base) for stringified numbers.  For example, a value of 16 indicates that
   * the value will be represented in hexadecimal.  This value may not be set
   * to values less than 2 or greater than 36.  <b>Default value:</b> 10.</p>
   *
   * @see #getRadix()
   * @see #setRadix(int)
   */
  protected int mnRadix = 10;

  /**
   * Retrieves Radix property.
   *
   * @return current value
   *
   * @see #mnRadix
   */
  public int getRadix() {return mnRadix;}

  /**
   * Updates Radix property.
   *
   * @param inRadix new value
   *
   * @see #mnRadix
   */
  public void setRadix(final int inRadix)
  {
    if (inRadix < 2 || inRadix > 36)
      throw new IllegalArgumentException("inRadix=" + inRadix);
    mnRadix = inRadix;
  }

  /////////////////
  //             //
  //  UpperCase  //
  //             //
  /////////////////

  /**
   * Upper case/lower case.
   *
   * <p><b>Details:</b> Property <code>UpperCase</code> determines whether upper
   * or lower case letters will be used when property <code>Radix</code> is
   * greater than 10.  If this property is <code>true</code>, upper case letters
   * will be used.  If this property is <code>false</code>, lower case letters
   * will be used.  The default value for this property is
   * <code>false</code>.</p>
   *
   * @see #getUpperCase()
   * @see #setUpperCase(boolean)
   */
  protected boolean mzUpperCase = false;

  /**
   * Retrieves UpperCase property.
   *
   * @return current value
   *
   * @see #mzUpperCase
   */
  public boolean getUpperCase() {return mzUpperCase;}

  /**
   * Updates UpperCase property.
   *
   * @param izUpperCase new value
   *
   * @see #mzUpperCase
   */
  public void setUpperCase(final boolean izUpperCase)
  {
      mzUpperCase = izUpperCase;
  }

  /////////////////
  //             //
  //  SigDigits  //
  //             //
  /////////////////

  /**
   * Significant digits.
   *
   * <p><b>Details:</b> Property <code>SigDigits</code> constrains the number of
   * significant digits to express in the output, so that the value will be
   * rounded if necessary.  Setting this value to 0 indicates that all digits
   * are considered significant.  This constraint may conflict with the
   * MaxRightDigits property in {@link RealFormatter}.  Default value: 0.</p>
   *
   * @see #getSigDigits()
   * @see #setSigDigits(int)
   */
  protected int mnSigDigits = 0;

  /**
   * Retrieves SigDigits property.
   *
   * @return current value
   *
   * @see #mnSigDigits
   */
  public int getSigDigits() {return mnSigDigits;}

  /**
   * Updates SigDigits property.
   *
   * @param inSigDigits new value
   *
   * @see #mnSigDigits
   */
  public void setSigDigits(final int inSigDigits)
  {
    if (inSigDigits < 0)
      throw new IllegalArgumentException("inSigDigits=" + inSigDigits);
    mnSigDigits = inSigDigits;
  }

  /////////////////
  //             //
  //  PosPrefix  //
  //             //
  /////////////////

  /**
   * Positive prefix.
   *
   * <p><b>Details:</b> Property <code>PosPrefix</code> is the string that
   * precedes positive values.  This property may not be <code>null</code>.
   * Default value: "".</p>
   *
   * @see #getPosPrefix()
   * @see #setPosPrefix(String)
   */
  protected String msPosPrefix = "";

  /**
   * Retrieves PosPrefix property.
   *
   * @return current value
   *
   * @see #msPosPrefix
   */
  public String getPosPrefix() {return msPosPrefix;}

  /**
   * Updates PosPrefix property.
   *
   * @param isPosPrefix new value
   *
   * @see #msPosPrefix
   */
  public void setPosPrefix(final String isPosPrefix)
  {
    if (isPosPrefix == null)
      throw new NullPointerException("isPosPrefix");
    msPosPrefix = isPosPrefix;
  }

  /////////////////
  //             //
  //  PosSuffix  //
  //             //
  /////////////////

  /**
   * Positive suffix.
   *
   * <p><b>Details:</b> Property <code>PosSuffix</code> is the string that
   * follows positive values.  This property may not be <code>null</code>.
   * Default value: "".</p>
   *
   * @see #getPosSuffix()
   * @see #setPosSuffix(String)
   */
  protected String msPosSuffix = "";

  /**
   * Retrieves PosSuffix property.
   *
   * @return current value
   *
   * @see #msPosSuffix
   */
  public String getPosSuffix() {return msPosSuffix;}

  /**
   * Updates PosSuffix property.
   *
   * @param isPosSuffix new value
   *
   * @see #msPosSuffix
   */
  public void setPosSuffix(final String isPosSuffix)
  {
    if (isPosSuffix == null)
      throw new NullPointerException("isPosSuffix");
    msPosSuffix = isPosSuffix;
  }

  //////////////////
  //              //
  //  ZeroPrefix  //
  //              //
  //////////////////

  /**
   * Zero prefix.
   *
   * <p><b>Details:</b> Property <code>ZeroPrefix</code> is the string that
   * precedes zero values.  This property may not be <code>null</code>.  Default
   * value: "".</p>
   *
   * @see #getZeroPrefix()
   * @see #setZeroPrefix(String)
   */
  protected String msZeroPrefix = "";

  /**
   * Retrieves ZeroPrefix property.
   *
   * @return current value
   *
   * @see #msZeroPrefix
   */
  public String getZeroPrefix() {return msZeroPrefix;}

  /**
   * Updates ZeroPrefix property.
   *
   * @param isZeroPrefix new value
   *
   * @see #msZeroPrefix
   */
  public void setZeroPrefix(final String isZeroPrefix)
  {
    if (isZeroPrefix == null)
      throw new NullPointerException("isZeroPrefix");
    msZeroPrefix = isZeroPrefix;
  }

  //////////////////
  //              //
  //  ZeroSuffix  //
  //              //
  //////////////////

  /**
   * Zero suffix.
   *
   * <p><b>Details:</b> Property <code>ZeroSuffix</code> is the string that
   * follows zero values.  This property may not be <code>null</code>.  Default
   * value: "".</p>
   *
   * @see #getZeroSuffix()
   * @see #setZeroSuffix(String)
   */
  protected String msZeroSuffix = "";

  /**
   * Retrieves ZeroSuffix property.
   *
   * @return current value
   *
   * @see #msZeroSuffix
   */
  public String getZeroSuffix() {return msZeroSuffix;}

  /**
   * Updates ZeroSuffix property.
   *
   * @param isZeroSuffix new value
   *
   * @see #msZeroSuffix
   */
  public void setZeroSuffix(final String isZeroSuffix)
  {
    if (isZeroSuffix == null)
      throw new NullPointerException("isZeroSuffix");
    msZeroSuffix = isZeroSuffix;
  }

  /////////////////
  //             //
  //  NegPrefix  //
  //             //
  /////////////////

  /**
   * Negative prefix.
   *
   * <p><b>Details:</b> Property <code>NegPrefix</code> is the string that
   * precedes negative values.  This property may not be <code>null</code>.
   * Default value: "-".</p>
   *
   * @see #getNegPrefix()
   * @see #setNegPrefix(String)
   */
  protected String msNegPrefix = "-";

  /**
   * Retrieves NegPrefix property.
   *
   * @return current value
   *
   * @see #msNegPrefix
   */
  public String getNegPrefix() {return msNegPrefix;}

  /**
   * Updates NegPrefix property.
   *
   * @param isNegPrefix new value
   *
   * @see #msNegPrefix
   */
  public void setNegPrefix(final String isNegPrefix)
  {
    if (isNegPrefix == null)
      throw new NullPointerException("isNegPrefix");
    msNegPrefix = isNegPrefix;
  }

  /////////////////
  //             //
  //  NegSuffix  //
  //             //
  /////////////////

  /**
   * Negative suffix.
   *
   * <p><b>Details:</b> Property <code>NegSuffix</code> is the string that
   * follows negative values.  This property may not be <code>null</code>.
   * Default value: "".</p>
   *
   * @see #getNegSuffix()
   * @see #setNegSuffix(String)
   */
  protected String msNegSuffix = "";

  /**
   * Retrieves NegSuffix property.
   *
   * @return current value
   *
   * @see #msNegSuffix
   */
  public String getNegSuffix() {return msNegSuffix;}

  /**
   * Updates NegSuffix property.
   *
   * @param isNegSuffix new value
   *
   * @see #msNegSuffix
   */
  public void setNegSuffix(final String isNegSuffix)
  {
    if (isNegSuffix == null)
      throw new NullPointerException("isNegSuffix");
    msNegSuffix = isNegSuffix;
  }

  /**
   * Default constructor.
   */
  protected NumberFormatter() {}

  /**
   * Adjusts significant digits.
   *
   * <p><b>Details:</b> This method adjusts the number of significant digits in
   * the given <code>BigDecimal</code> and returns the adjusted value.</p>
   *
   * <p>This method is a temporary hack to overcome a limitation in
   * <code>BigDecimal</code>.  A feature request has been submitted to the Java
   * Bug Parade and it has been address.  This method should be reimplemented
   * using the latest methods available in <code>BigDecimal</code>.</p>
   *
   * @param ipValue value to adjust
   * @param inSigDigits number of significant digits
   * @return adjusted value
   */
  static final BigDecimal setSignificantDigits(BigDecimal ipValue, final int inSigDigits)
  {
    if (inSigDigits <= 0)
      throw new IllegalArgumentException("inSigDigits=" + inSigDigits);
    final int vnScale = ipValue.scale();
    ipValue = ipValue.movePointRight(vnScale);
    final String vsString = ipValue.toString();
    int vnSize = vsString.length();
    if (vsString.charAt(vnSize - 1) == '.')
      -- vnSize;
    if (vsString.charAt(0) == '-')
      -- vnSize;
    ipValue = ipValue.movePointLeft(vnSize);
    ipValue = ipValue.setScale(inSigDigits, BigDecimal.ROUND_HALF_UP);
    return ipValue.movePointRight(vnSize - vnScale);
  }

}

