package com.sharkysoft.printf.engine;

import com.sharkysoft.math.MantissaExponent;
import com.sharkysoft.string.StringToolbox;
import com.sharkysoft.util.UnreachableCodeException;
import java.math.BigDecimal;

/**
 * Formats real numbers.
 *
 * <p><b>Details:</b> A <code>RealFormatter</code> formats numbers according to 
 * criteria selected by the user.  The user can control various formatting 
 * parameters, such as field width, justification style (left-<wbr>align, 
 * right-<wbr>align, center, etc.), precision, etc.</p>
 *
 * <p>Since applications that format real numbers frequently format more than
 * one real number with the same style, this class is designed to "remember" the
 * currently selected formatting parameters so that they can be reused with each
 * format call.  This approach reduces the amount of parameters which must be
 * passed during the format calls and results in more efficient execution.</p>
 *
 * <blockquote class="example">
 *
 *   <p><b>Example:</b> Output the contents of a <code>float</code> array called
 *   <var>vals</var>, one per line, right-justified in a 25-character-wide
 *   field.  Round each float to two decimal places, but always show the two
 *   decimal places even if they are zeros.</p>
 *
    *<blockquote><pre>
		   *float[] vals = {1.0F, (float) Math.PI, -22};
		   *RealFormatter formatter = new RealFormatter();
		   *formatter.setFieldWidth(25);
		   *formatter.setPadChar(':'); // non-space chosen for emphasis
		   *formatter.setAlignment(AlignmentMode.gpRight);
		   *formatter.setMinRightDigits(2);
		   *formatter.setMaxRightDigits(2);
		   *for (int i = 0; i &lt; vals.length; ++ i)
		   *  System.out.println(formatter.format(vals[i]));
    *</pre></blockquote>
 *
 * <p>This produces the following output:</p>
 *
  *<blockquote><pre>
   *:::::::::::::::::::::1.00
   *:::::::::::::::::::::3.14
   *:::::::::::::::::::-22.00
  *</pre></blockquote>
 *
 * </blockquote>
 *
 * <p>Formatting parameters can be updated at any time, and doing so will change
 * the way text is formatted afterwards.  Methods in this class only support
 * formatting for <code>radix</code>=10.  For a detailed description of all
 * user-<wbr>controllable parameters, see the documentation below and in the
 * {@linkplain NumberFormatter superclass}.</p>
 */
public class RealFormatter extends NumberFormatter
{

  /////////////////////////
  //                     //
  //  ExponentFormatter  //
  //                     //
  /////////////////////////

  /**
   * Exponent mpFormatter for scientific notation.
   *
   * <p><b>Details:</b> Property <code>ExponentFormatter</code> is the configurable
   * {@link IntegerFormatter} used to render exponent integers when real values
   * are rendered in scientific notation.</p>
   *
   * <p><b>Initial configuration:</b>
   * <code>MinDigits</code>=2,
   * <code>ZeroPrefix</code>="E",
   * <code>PosPrefix</code>="E+",
   * <code>NegPrefix</code>="E-".</p>
   *
   * @see #getExponentFormatter()
   */
  protected final IntegerFormatter mpExponentFormatter = new IntegerFormatter();

  /**
   * Retrieves ExponentFormatter property.
   *
   * @return current value
   *
   * @see #mpExponentFormatter
   */
  public IntegerFormatter getExponentFormatter()
  {
    return mpExponentFormatter;
  }

  /////////////////////
  //                 //
  //  MaxLeftDigits  //
  //                 //
  /////////////////////

  /**
   * Maximum digits left of decimal point.
   *
   * <p><b>Details:</b> Property <code>MaxLeftDigits</code> is the maximum
   * number of digits to render left of the decimal point.  The value will wrap
   * like an odometer if it is too large.  <b>Default value:</b>
   * <code>Integer.MAX_VALUE</code>.</p>
   *
   * @see #getMaxLeftDigits()
   * @see #setMaxLeftDigits(int)
   */
  protected int mnMaxLeftDigits = Integer.MAX_VALUE;

  /**
   * Retrieves MaxLeftDigits property.
   *
   * @return current value
   *
   * @see #mnMaxLeftDigits
   */
  public int getMaxLeftDigits()
  {
    return mnMaxLeftDigits;
  }

  /**
   * Updates MaxLeftDigits property.
   *
   * <p><b>Details:</b> This setter updates the <code>MaxLeftDigits</code>
   * property.  The new value may not be smaller than zero.  If the new value is
   * smaller than <code>MinLeftDigits</code>, then <code>MinLeftDigits</code> will
   * be set to this new value as well.</p>
   *
   * @param inMaxLeftDigits new value
   *
   * @see #mnMaxLeftDigits
   */
  public void setMaxLeftDigits(final int inMaxLeftDigits)
  {
    if (inMaxLeftDigits < 0)
      throw new IllegalArgumentException("inMaxLeftDigits=" + inMaxLeftDigits);
    if (inMaxLeftDigits < mnMinLeftDigits)
      mnMinLeftDigits = inMaxLeftDigits;
    mnMaxLeftDigits = inMaxLeftDigits;
  }

  /////////////////////
  //                 //
  //  MinLeftDigits  //
  //                 //
  /////////////////////

  /**
   * Minimum digits left of decimal point.
   *
   * <p><b>Details:</b> Property <code>MinLeftDigits</code> is the minimum
   * number of digits to render left of the decimal point.  If the value isn't
   * large enough, the value will be padded with the left pad digit
   * ({@link #mcLeftPadDigit LeftPadDigit}).  <b>Default value:</b> 1.</p>
   *
   * @see #getMinLeftDigits()
   * @see #setMinLeftDigits(int)
   */
  protected int mnMinLeftDigits = 1;

  /**
   * Retrieves MinLeftDigits property.
   *
   * @return current value
   *
   * @see #mnMinLeftDigits
   */
  public int getMinLeftDigits()
  {
    return mnMinLeftDigits;
  }

  /**
   * Updates MinLeftDigits property.
   *
   * <p><b>Details:</b> This setter updates the <code>MinLeftDigits</code>
   * property.  The new value may not be smaller than zero.  If the new value is
   * larger than <code>MaxLeftDigits</code>, then <code>MaxLeftDigits</code>
   * will be set to the new value as well.</p>
   *
   * @param inMinLeftDigits new value
   *
   * @see #mnMinLeftDigits
   */
  public void setMinLeftDigits(final int inMinLeftDigits)
  {
    if (inMinLeftDigits < 0)
      throw new IllegalArgumentException("inMinLeftDigits=" + inMinLeftDigits);
    if (inMinLeftDigits > mnMaxLeftDigits)
      mnMaxLeftDigits = inMinLeftDigits;
    mnMinLeftDigits = inMinLeftDigits;
  }

  ////////////////////
  //                //
  //  LeftPadDigit  //
  //                //
  ////////////////////

  /**
   * Left pad digit.
   *
   * <p><b>Details:</b> Property <code>LeftPadDigit</code> determines the digit
   * or character to use for left-padding numbers.  This character will be used
   * only when the value of <code>MinLeftDigits</code> requires it.  If this
   * property is set to 0, the space character will be used, but the prefix
   * string will appear <em>between</em> the pad digits and the actual number.
   * Otherwise, the padding will appear between the prefix string and the
   * number.  <b>Default value:</b> '0'.</p>
   *
   * @see #getLeftPadDigit()
   * @see #setLeftPadDigit(char)
   */
  protected char mcLeftPadDigit = '0';

  /**
   * Retrieves LeftPadDigit property.
   *
   * @return current value
   *
   * @see #mcLeftPadDigit
   */
  public char getLeftPadDigit()
  {
    return mcLeftPadDigit;
  }

  /**
   * Updates LeftPadDigit property.
   *
   * @param icLeftPadDigit new value
   *
   * @see #mcLeftPadDigit
   */
  public void setLeftPadDigit(final char icLeftPadDigit)
  {
    mcLeftPadDigit = icLeftPadDigit;
  }

  ////////////////////
  //                //
  //  ShowDecPoint  //
  //                //
  ////////////////////

  /**
   * Show decimal point when optional.
   *
   * <p><b>Details:</b> Property <code>ShowDecPoint</code> determines whether or
   * not the decimal point will be included when it is possible render the
   * number without it.  <b>Default value: <code>false</code></b>.</p>
   *
   * @see #getShowDecPoint()
   * @see #setShowDecPoint(boolean)
   */
  protected boolean mzShowDecPoint = false;

  /**
   * Retrieves ShowDecPoint property.
   *
   * @return current value
   *
   * @see #mzShowDecPoint
   */
  public boolean getShowDecPoint()
  {
    return mzShowDecPoint;
  }

  /**
   * Updates ShowDecPoint property.
   *
   * @param izShowDecPoint
   *
   * @see #mzShowDecPoint
   */
  public void setShowDecPoint(final boolean izShowDecPoint)
  {
    mzShowDecPoint = izShowDecPoint;
  }

  //////////////////////
  //                  //
  //  MinRightDigits  //
  //                  //
  //////////////////////

  /**
   * Minimum digits right of the decimal point.
   *
   * <p><b>Details:</b> Property <code>MinRightDigits</code> determines the
   * minimum number of digits that will be rendered right of the decimal point.  If
   * the fractional portion of the number isn't long enough to fill out the minimum
   * number of digits, the extra places
   * will be set to <code>RightPadDigit</code>.  <b>Default value:</b> 0.
   *
   * @see #getMinRightDigits()
   * @see #setMinRightDigits(int)
   */
  protected int mnMinRightDigits = 0;

  /**
   * Retrieves MinRightDigits property.
   *
   * @return current value
   *
   * @see #mnMinRightDigits
   */
  public int getMinRightDigits()
  {
    return mnMinRightDigits;
  }

  /**
   * Updates MinRightDigits property.
   *
   * @param inMinRightDigits new value
   *
   * @see #mnMinRightDigits
   */
  public void setMinRightDigits(final int inMinRightDigits)
  {
    if (inMinRightDigits < 0)
      throw new IllegalArgumentException("inMinRightDigits=" + inMinRightDigits);
    if (inMinRightDigits > mnMaxRightDigits)
      mnMaxRightDigits = inMinRightDigits;
    mnMinRightDigits = inMinRightDigits;
  }

  /////////////////////
  //                 //
  //  RightPadDigit  //
  //                 //
  /////////////////////

  /**
   * Right pad digit.
   *
   * <p><b>Details:</b> Property <code>RightPadDigit</code> determines the digit
   * or character to use for right-<wbr>padding numbers.  This character will be used
   * only when the
   * value of <code>MinRightDigits</code> requires it.  If this property is set
   * to 0, the space
   * character will be used, but the suffix string will appear <em>between</em>
   * the pad digits and the actual number.  Otherwise, the padding will appear
   * between the suffix string and the number.  <b>Default value:</b> '0'.</p>
   *
   * @see #getRightPadDigit()
   * @see #setRightPadDigit(char)
   */
  protected char mcRightPadDigit = '0';

  /**
   * Retrieves RightPadDigit property.
   *
   * @return current value
   *
   * @see #mcRightPadDigit
   */
  public char getRightPadDigit()
  {
    return mcRightPadDigit;
  }

  /**
   * Updates RightPadDigit property.
   *
   * @param icRightPadDigit new value
   *
   * @see #mcRightPadDigit
   */
  public void setRightPadDigit(final char icRightPadDigit)
  {
    mcRightPadDigit = icRightPadDigit;
  }

  //////////////////////
  //                  //
  //  MaxRightDigits  //
  //                  //
  //////////////////////

  /**
   * Maximum digits right of decimal point.
   *
   * <p><b>Details:</b> Property <code>MaxRightDigits</code> determines the
   * maximum number of digits to show on the right of the decimal point.  If the
   * fractional portion of the number is "longer" than this number of places,
   * the value will be rounded.  If, for a particular value being rendered, the
   * configuration of <code>MaxRightDigits</code> and <code>SigDigits</code> is
   * incompatible, the rendered value will rounded to the appropriate number of
   * significant digits, and then the trailing zeros will be trimmed, if there
   * are any, until the <code>MaxRightDigits</code> constraint is satisfied, or
   * until there are no more trailing zeros to trim.  <b>Default value:</b>
   * <code>Integer.MAX_VALUE</code>.</p>
   *
   * @see #getMaxRightDigits()
   * @see #setMaxRightDigits(int)
   */
  protected int mnMaxRightDigits = Integer.MAX_VALUE;

  /**
   * Retrieves MaxRightDigits property.
   *
   * @return current value
   *
   * @see #mnMaxRightDigits
   */
  public int getMaxRightDigits()
  {
    return mnMaxRightDigits;
  }

  /**
   * Updates MaxRightDigits property.
   *
   * @param inMaxRightDigits new value
   *
   * @see #mnMaxRightDigits
   */
  public void setMaxRightDigits(final int inMaxRightDigits)
  {
    if (inMaxRightDigits < 0)
      throw new IllegalArgumentException("inMaxRightDigits=" + inMaxRightDigits);
    if (inMaxRightDigits < mnMinRightDigits)
      mnMinRightDigits = inMaxRightDigits;
    mnMaxRightDigits = inMaxRightDigits;
  }

  ////////////////////
  //                //
  //  constructors  //
  //                //
  ////////////////////

  /**
   * Default constructor.
   *
   * <p><b>Details:</b> This constructor initializes a new instance with the
   * default value for each property, as specified in the property
   * documentation.</p>
   */
  public RealFormatter()
  {
    super();
    mpExponentFormatter.msZeroPrefix = "E";
    mpExponentFormatter.msNegPrefix  = "E-";
    mpExponentFormatter.msPosPrefix  = "E+";
    mpExponentFormatter.mnMinDigits = 2;
  }

  //////////////
  //          //
  //  format  //
  //          //
  //////////////

  private final String format(BigDecimal ipValue, final String isExponentSuffix)
  {
    if (mnRadix != 10)
      throw new IllegalArgumentException("mnRadix=" + mnRadix);
    // Choose the prefix and suffix:
    final String vsPrefix, vsSuffix;
    switch (ipValue.signum())
    {
    case -1:
      vsPrefix = msNegPrefix;
      vsSuffix = msNegSuffix;
      ipValue = ipValue.negate();
      break;
    case 0:
      vsPrefix = msZeroPrefix;
      vsSuffix = msZeroSuffix;
      break;
    case +1:
      vsPrefix = msPosPrefix;
      vsSuffix = msPosSuffix;
      break;
    default:
      throw new UnreachableCodeException();
    }
    final String vsString;
    // If significant digits is specified, adjust the BigDecimal's significant
    // digits by rounding:
    if (mnSigDigits > 0)
    {
      ipValue = setSignificantDigits(ipValue, mnSigDigits);
      final int vnTrimAmt = ipValue.scale() - mnMaxRightDigits;
      if (vnTrimAmt > 0)
      {
        // We're in here because the sig_digits setting is incompatible with the
        // max_right_digits setting for this particular value.  We will try to
        // trim the current value's trailing zeros so that max_right_digits is
        // satisfied.
        final StringBuffer vpBuff = new StringBuffer(ipValue.toString());
        int vnLast = vpBuff.length() - 1;
        for (int vnI = 0; vnI < vnTrimAmt; ++ vnI)
        {
          if (vpBuff.charAt(vnLast) != '0')
            break;
          -- vnLast;
        }
        vpBuff.setLength(vnLast + 1);
        vsString = vpBuff.toString();
      }
      else
        vsString = ipValue.toString();
    }
    else
    {
      // Round the number so that it does not have more than max_right_digits
      // fractional digits.  Round with impunity since there is no sig_digits
      // setting.
      if (ipValue.scale() > mnMaxRightDigits)
        ipValue = ipValue.setScale(mnMaxRightDigits, BigDecimal.ROUND_HALF_UP);
      vsString = ipValue.toString();
    }
    // Separate the whole and fractional parts of the value:
    final String vsLeftDigits, vsRightDigits;
    {
      int vnDpPos = vsString.indexOf('.');
      if (vnDpPos == -1)
      {
        vsLeftDigits = vsString;
        vsRightDigits = "";
      }
      else
      {
        int vnLds = vnDpPos - mnMaxLeftDigits;
        if (vnLds < 0)
          vnLds = 0;
        vsLeftDigits = vsString.substring(vnLds, vnDpPos);
        ++ vnDpPos;
        if (vnDpPos == vsString.length())
          vsRightDigits = "";
        else
          vsRightDigits = vsString.substring(vnDpPos);
      }
    }
    // Make left extender:
    final String vsLeftExtender;
    {
      final int vnExtra = mnMinLeftDigits - vsLeftDigits.length();
      if (vnExtra > 0)
      {
        if (mcLeftPadDigit != 0)
          vsLeftExtender = StringToolbox.repeat(mcLeftPadDigit, vnExtra);
        else
          vsLeftExtender = StringToolbox.repeat(' ', vnExtra);
      }
      else
        vsLeftExtender = "";
    }
    // Assume that no decimal point will be needed until we discover otherwise:
    boolean vzNeedDecPoint = mzShowDecPoint;
    // Make right extender:
    final String vsRightExtender;
    {
      final int vnLength = vsRightDigits.length();
      if (vnLength > 0)
        vzNeedDecPoint = true;
      final int vnExtra = mnMinRightDigits - vsRightDigits.length();
      if (vnExtra > 0)
      {
        if (mcRightPadDigit != 0)
        {
          vsRightExtender = StringToolbox.repeat(mcRightPadDigit, vnExtra);
          if (mcRightPadDigit != ' ')
            vzNeedDecPoint = true;
        }
        else
          vsRightExtender = StringToolbox.repeat(' ', vnExtra);
      }
      else
        vsRightExtender = "";
    }
    final StringBuffer vpLeftSide = new StringBuffer();
    final StringBuffer vpRightSide = new StringBuffer();
    // Build leader:
    if (mnAlignment == AlignmentMode.FULL)
    {
      vpLeftSide.append(vsPrefix);
      vpRightSide.append(vsLeftExtender);
    }
    else
    {
      if (mcLeftPadDigit != 0)
      {
        vpRightSide.append(vsPrefix);
        vpRightSide.append(vsLeftExtender);
      }
      else
      {
        vpRightSide.append(vsLeftExtender);
        vpRightSide.append(vsPrefix);
      }
    }
    // Append actual number:
    vpRightSide.append(vsLeftDigits);
    if (vzNeedDecPoint)
      vpRightSide.append('.');
    vpRightSide.append(vsRightDigits);
    // Append trailer:
    if (mcRightPadDigit != 0)
    {
      vpRightSide.append(vsRightExtender);
      vpRightSide.append(isExponentSuffix).append(vsSuffix);
    }
    else
    {
      vpRightSide.append(isExponentSuffix).append(vsSuffix);
      vpRightSide.append(vsRightExtender);
    }
    // Concatenate the two sides together and finish formatting:
    return format(vpLeftSide.toString(), vpRightSide.toString());
  }

  /**
   * Formats real value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given real value
   * (<code>ipValue</code>) into the format specified by the current property
   * configuration.  The value is rendered into scientific notation.</p>
   *
   * @param ipValue the real value
   * @return the formatted value
   */
  public final String format(final MantissaExponent ipValue)
  {
  	final BigDecimal vpMantissa = ipValue.getMantissa();
  	if (vpMantissa == null)
  		switch (ipValue.getExponent())
  		{
  		case MantissaExponent.NAN:
				return formatSpecial(msPosPrefix + "nan");
			case MantissaExponent.NEG_INF:
				return formatSpecial(msNegPrefix + "inf");
			case MantissaExponent.POS_INF:
				return formatSpecial(msZeroPrefix + "inf");
			default:
				throw new UnreachableCodeException("getExponent returned spurious value");
  		}
    return format(vpMantissa, mpExponentFormatter.format(ipValue.getExponent()));
  }

  /**
   * Formats real value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given real value
   * (<code>ipValue</code>) into the format specified by the current property
   * configuration.</p>
   *
   * @param ipValue the real value
   * @return the formatted value
   */
  public final String format(final BigDecimal ipValue)
  {
    return format(ipValue, "");
  }

	private String formatSpecial(final String isSpecial)
	{
		switch (mnAlignment)
		{
		case AlignmentMode.FULL:
			return format("", isSpecial, ' ');
		default:
			return format(isSpecial, ' ');
		}
	}
	
  /**
   * Formats real value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given real value
   * (<code>ifValue</code>) into the format specified by the current property
   * configuration.</p>
   *
   * @param ifValue the real value
   * @return the formatted value
   */
  public final String format(final float ifValue)
  {
		if (ifValue == Float.NEGATIVE_INFINITY)
			return formatSpecial(msNegPrefix + "inf");
		if (ifValue == Float.POSITIVE_INFINITY)
			return formatSpecial(msPosPrefix + "inf");
  	if (Float.isNaN(ifValue))
			return formatSpecial(msZeroPrefix + "nan");
    return format(new BigDecimal(ifValue));
  }

  /**
   * Formats real value.
   *
   * <p><b>Details:</b> <code>format</code> renders the given real value
   * (<code>idValue</code>) into the format specified by the current property
   * configuration.</p>
   *
   * @param idValue the real value
   * @return the formatted value
   */
  public final String format(final double idValue)
  {
		if (idValue == Double.NEGATIVE_INFINITY)
			return formatSpecial(msNegPrefix + "inf");
		if (idValue == Double.POSITIVE_INFINITY)
			return formatSpecial(msPosPrefix + "inf");
		if (Double.isNaN(idValue))
			return formatSpecial(msZeroPrefix + "nan");
    return format(new BigDecimal(idValue));
  }

}

