package com.sharkysoft.printf.engine;

import com.sharkysoft.string.StringToolbox;
import com.sharkysoft.util.UnreachableCodeException;

/**
 * Justifies text in field.
 *
 * <p><b>Details:</b> A <code>StringFormatter</code> formats arbitrary length
 * strings into a fixed-<wbr>length field according to specifiable formatting
 * criteria.</p>
 *
 * <p>To justify a string in a field, create an instance of
 * <code>StringFormatter</code>, configure the formatting properties, and then
 * process the string through the justifier using
 * {@link #format(String) format}.</p>
 *
 * <p><b>Example:</b> Output the contents of <code>vasNames</code>, an array of
 * strings, with one element per line, right-<wbr>justified in a
 * 50-<wbr>character field.  For strings shorter than 50 characters, fill in the
 * open space on the left with an underscore character.</p>
 *
  *<blockquote><pre>
    *String[] vasNames = {"apple", "banana", "cherry"};
    *StringFormatter vpJustifier = new StringFormatter();
    *vpJustifier.setFieldWidth(50);
    *vpJustifier.setAlignment(AlignmentMode.gpRight);
    *vpJustifier.setPadChar('_');
    *for (int vnI = 0; vnI &lt; vasNames.length; ++ i)
    *  System.out.println(vpJustifier.format(vasNames[i]));
  *</pre></blockquote>
 *
 * <p>The output of this code is:</p>
 *
    *<blockquote><pre>
     *_____________________________________________apple
     *____________________________________________banana
     *____________________________________________cherry
   *</pre></blockquote>
 *
 * @author Sharky
 */
public class StringFormatter
{

  //////////////////
  //              //
  //  FieldWidth  //
  //              //
  //////////////////

  /**
   * Field width.
   *
   * <p><b>Details:</p> Property <code>FieldWidth</code> is the width of the
   * field into which text will be formatted.</p>
   *
   * <p><b>Default value:</b> <code>0</code>.</p>
   *
   * @see #getFieldWidth()
   * @see #setFieldWidth(int)
   */
  protected int mnFieldWidth = 0;

  /**
   * Retrieves FieldWidth property.
   *
   * @return current value
   *
   * @see #mnFieldWidth
   */
  public int getFieldWidth() {return mnFieldWidth;}

  /**
   * Updates FieldWidth property.
   *
   * @param inFieldWidth new value
   *
   * @see #mnFieldWidth
   */
  public void setFieldWidth(final int inFieldWidth)
  {
    if (inFieldWidth < 0)
      throw new IllegalArgumentException("inFieldWidth=" + inFieldWidth);
    mnFieldWidth = inFieldWidth;
  }

  /////////////////
  //             //
  //  Alignment  //
  //             //
  /////////////////

  /**
   * Alignment mode.
   *
   * <p><b>Details:</b> Property <code>Alignment</code> controls the positioning
   * of text within the field.  Available alignment options include left
   * justification, right justification, full justification, and centering.</p>
   *
   * <p><b>Default value:</b>
   * {@link AlignmentMode#gpLeft AlignmentMode.gpLeft}.</p>
   *
   * @see #getAlignment()
   * @see #setAlignment(AlignmentMode)
   */
  protected int mnAlignment = AlignmentMode.LEFT;

  /**
   * Retrieves Alignment property.
   *
   * @return current value
   *
   * @see #mnAlignment
   */
  public AlignmentMode getAlignment()
  {
      return AlignmentMode.forInt(mnAlignment);
  }

  /**
   * Updates Alignment property.
   *
   * @param ipAlignment new value
   *
   * @see #mnAlignment
   */
  public void setAlignment(final AlignmentMode ipAlignment)
  {
    mnAlignment = ipAlignment.toInt();
  }

  ///////////////
  //           //
  //  PadChar  //
  //           //
  ///////////////

  /**
   * Padding character.
   *
   * <p><b>Details:</b> Property <code>PadChar</code> is the character used to
   * fill the unused portion of the field when the text is shorter than the
   * field.</p>
   *
   * <p><b>Default value:</b> '&nbsp;' (space character).</p>
   *
   * @see #getPadChar()
   * @see #setPadChar(char)
   */
  protected char mcPadChar = ' ';

  /**
   * Retrieves PadChar property.
   *
   * @return current value
   *
   * @see #mcPadChar
   */
  public char getPadChar() {return mcPadChar;}

  /**
   * Updates PadChar property.
   *
   * @param icPadChar new value
   *
   * @see #mcPadChar
   */
  public void setPadChar(final char icPadChar) {mcPadChar = icPadChar;}

  ////////////////
  //            //
  //  Cropping  //
  //            //
  ////////////////

  /**
   * Cropping mode.
   *
   * <p><b>Details:</b> Property <code>Cropping</code> controls whether and how
   * text is compressed if it is larger than the field into which it is
   * formatted.  The head, tail, or center portion of the text may be cropped to
   * ensure that the formatted text fits within the field.  Or, cropping may be
   * prohibited, so that long text actually overflows the field.</p>
   *
   * <p><b>Default value:</b>
   * {@link CroppingMode#gpNone CroppingMode.gpNone}.</p>
   *
   * @see #getCropping()
   * @see #setCropping(CroppingMode)
   */
  protected int mnCropping = CroppingMode.NONE;

  /**
   * Retrieves Cropping property.
   *
   * @return current value
   *
   * @see #mnCropping
   */
  public CroppingMode getCropping()
  {
      return CroppingMode.forInt(mnCropping);
  }

  /**
   * Updates Cropping property.
   *
   * @param ipCropping new value
   *
   * @see #mnCropping
   */
  public void setCropping(final CroppingMode ipCropping)
  {
    mnCropping = ipCropping.toInt();
  }

  /**
   * Default constructor.
   */
  public StringFormatter() {}

  /**
   * Crops text.
   *
   * <p><b>Details:</b> <code>crop</code> crops the supplied string according to
   * the currently configured properties.  <code>isText</code> must not be
   * <code>null</code>.</p>
   *
   * @param isText the string to crop
   * @return the cropped string
   */
  private final String crop(final String isText)
  {
    final int vnPadSize = mnFieldWidth - isText.length();
    if (vnPadSize <= 0)
      switch (mnCropping)
      {
      case CroppingMode.NONE:
        return isText;
      case CroppingMode.LEFT:
        return isText.substring(- vnPadSize);
      case CroppingMode.RIGHT:
        return isText.substring(0, mnFieldWidth);
      case CroppingMode.MIDDLE:
        return isText.substring(- vnPadSize / 2, - vnPadSize / 2 + mnFieldWidth);
      default:
        throw new UnreachableCodeException();
      }
    return isText;
  }

  /**
   * Justifies text.
   *
   * <p><b>Details:</b> <code>format</code> formats the supplied string
   * according to currently configured properties.</p>
   *
   * <p>If <code>Alignment</code> is set to <code>FULL</code>, then the
   * split-<wbr>string format method ({@link #format(String, String)}) should be
   * used instead.  This is because the format method with only one string
   * parameter does not know where the filler characters should be inserted.
   * Eventually, this method may be upgraded to support automatic expansion of
   * whitespace characters within the string for full justification.</p>
   *
   * @param isText the text to justify
   * @return the formatted string
   */
  public final String format(final String isText)
  {
  	return format(isText, mcPadChar);
  }

	final String format(final String isText, final char icPad)
	{
		int vnPadSize = mnFieldWidth - isText.length();
		if (vnPadSize <= 0)
			return crop(isText);
		final String vsPadding = StringToolbox.repeat(icPad, vnPadSize);
		switch (mnAlignment)
		{
		case AlignmentMode.RIGHT:
			return vsPadding + isText;
		case AlignmentMode.LEFT:
			return isText + vsPadding;
		case AlignmentMode.CENTER:
			vnPadSize /= 2;
			return vsPadding.substring(0, vnPadSize) + isText + vsPadding.substring(vnPadSize);
		case AlignmentMode.FULL:
			throw new UnsupportedOperationException("full justification");
		default:
			throw new UnreachableCodeException();
		}
	}

  /**
   * Justifies text.
   *
   * <p><b>Details:</b> <code>format</code> justifies a split string according
   * to the currently configured properties.  If <code>Alignment</code> is set
   * to <code>FULL</code>, then the prefix is left-<wbr>aligned and the suffix
   * is right-<wbr>aligned, and the portion between them is filled with the
   * padding character.  Otherwise, the strings are simply concatenated together
   * and formatted as in {@link #format(String)}.</p>
   *
   * @param isPrefix the left portion of the string
   * @param isSuffix the right portion of the string
   * @return the formatted string
   */
  public final String format(final String isPrefix, final String isSuffix)
  {
  	return format(isPrefix, isSuffix, mcPadChar);
  }

	final String format(final String isPrefix, final String isSuffix, final char icPad)
	{
		if (mnAlignment != AlignmentMode.FULL)
			return format(isPrefix + isSuffix, icPad);
		final int vnPadSize = mnFieldWidth - (isPrefix.length() + isSuffix.length());
		if (vnPadSize <= 0)
			return crop(isPrefix + isSuffix);
		final String vsPadding = StringToolbox.repeat(icPad, vnPadSize);
		return isPrefix + vsPadding + isSuffix;
	}

  /**
   * Generates debug output.
   *
   * <p><b>Details:</b> <code>toString</code> generates a string representation
   * of this instance and its properties.  This output is useful only for
   * debugging.</p>
   *
   * @return string representation
   */
  public String toString()
  {
    return "StringFormatter {"
        + "FieldWidth: " + mnFieldWidth
        + ", Alignment: " + AlignmentMode.toString(mnAlignment)
        + ", PadChar: (char) " + (int) mcPadChar
        + ", Cropping:" + CroppingMode.toString(mnCropping)
        + "}";
  }

}

