///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//  Notice to licensees:                                                     //
//                                                                           //
//  This source code is the exclusive, proprietary intellectual property of  //
//  Sharkysoft (sharkysoft.com).  You may view this source code as a         //
//  supplement to other product documentation, but you may not distribute    //
//  it or use it for any other purpose without written consent from          //
//  Sharkysoft.                                                              //
//                                                                           //
//  You are permitted to modify and recompile this source code, but you may  //
//  not remove this notice.  If you add features to or fix errors in this    //
//  code, please consider sharing your changes with Sharkysoft for possible  //
//  incorporation into future releases of the product.  Thanks!              //
//                                                                           //
//  For more information about Sharkysoft products and services, please      //
//  visit Sharkysoft on the web at                                           //
//                                                                           //
//       http://sharkysoft.com/                                              //
//                                                                           //
//  Thank you for using Lava!                                                //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

package com.sharkysoft.util;

/**
 * Indicates an unfinished execution path.
 *
 * <p><b>Details:</b> A <code>NotImplementedException</code> is thrown when a
 * method is called, a case invoked, or a feature is requested that has not yet
 * been implemented.  This class is the extreme programmer's best friend; it
 * allows the developer to compile and execute half-<wbr>baked code while
 * minimizing the risk of forgetting to finish it later.  More importantly, this
 * class allows you to completely design a class without implementing any of it.
 * To do this, simply include the statement</p>
 *
 * <blockquote><code>
 *   throw new NotImplementedException();
 * </blockquote></ul>
 *
 * <p>in any method or execution path that you'd like to finish later.  The
 * compiler will let you compile the code without forcing you to insert dummy
 * return values or "TO<!-- fool IDE -->DO" comments that you might forget about later.</p>
 *
 * <blockquote>
 *
 *   <p><b>Example 1:</b></p>
 *
 * <!-- The arrangement of the asterisks has special significance in javadoc
 * comments.  Please ask Sharky for details if you are curious. -->
 *
     <pre>
      *if (bReadyToRock)
      *{ do_stuff();
      *  do_more_stuff();
      *}
      *else
      *  throw new NotImplementedException();
     </pre>
 *
 * </blockquote>
 *
 * <p>In Example 1, a single execution path has been left unimplemented because
 * the programmer is not yet concerned about it.  He'll get to it later, and if
 * he forget's he'll be reminded when it's needed.</p>
 *
 * <blockquote>
 *
 *   <p><b>Example 2:</b></p>
 *
     <pre>
      *public static float[] findPolynomialRoots(Polynomial ipPolynomial)
      *{ // I'm putting this declaration here so that the rest of my code will
      *  // compile, but I'll implement the body later or ask my partner to do
      *  //  it.
      *  throw new NotImplementedException();
      *}
     </pre>
 *
 * </blockquote>
 *
 * <p>In Example 2, a method is stubbed out because the developer knows he's
 * going to need it, but he doesn't want to focus on its implementation right
 * now.  In situations like this, you may be tempted to stub out the method and
 * return a dummy value, such as <code>null</code> or 0.  Don't!  Doing so is
 * asking for trouble!  You might forget about it and wind up with a
 * difficult-<wbr>to-<wbr>trace bug!  Just use
 * <code>NotImplementedException</code> instead, and rest confident that the JVM
 * will remind you if your forgotten method ever gets executed.  In fact, you
 * won't just be reminded, but you'll also be pointed to the class, method, and
 * line number!</p>
 *
 * <blockquote>
 *
 *   <p><b>Example 3:</b></p>
 *
     <pre>
      *public void actionPerformed(ActionEvent ipEvent)
      *{ // We're safe doing nothing for now, but eventually we'll need to
      *  // implement this.
      *  NotImplementedException.trace("warning: button response not implemented");
      *}
     </pre>
 *
 * </blockquote>
 *
 * <p>In Example 3, a button's action event handler has not yet been
 * implemented.  At this stage in the application development, however, the
 * programmer has decided that this is a problem.  Rather than stopping the show
 * with an exception, the developer has decided to simply let
 * <code>NotImplementedException</code> issue a reminder that the handler has
 * not yet been implemented.</p>
 *
 * <p>Before deploying the final build of an application, the developer should
 * search all source files for instances of the identifier
 * "<code>NotImplementedException</code>" to be sure that application is indeed
 * complete.</p>
 *
 * @since 1997
 * @version 1999.04.19
 */
public class NotImplementedException extends RuntimeException
{

	/**
	 * Initializes without detail.
	 *
	 * <p><b>Details:</b>  This default constructor initializes a new
	 * <code>NotImplementedException</code> without a detail message.</p>
	 */
	public NotImplementedException()
	{
		super();
	}

	/**
	 * Initializes with detail.
	 *
	 * <p><b>Details:</b>  This constructor initializes a new
	 * <code>NotImplementedException</code> with the given detail message.</p>
	 *
	 * @param isDetail the messsage
	 */
	public NotImplementedException(String isDetail)
	{
		super(isDetail);
	}

	/**
	 * Prints stack trace of unimplemented execution path.
	 *
	 * <p><b>Details:</b>  <code>trace</code> instantiates a new
	 * <code>NotImplementedException</code> without throwing it, and then outputs
	 * its stack trace to <code>System.err</code>.  This is useful for tracking
	 * unfinished code paths which must be allowed to operate, albeit temporarily,
	 * in their unfinished state.  The provided detail message is included in the
	 * stack trace.</p>
	 *
	 * @param isDetail detail message
	 * @see #trace()
	 * @since 1999.06.04
	 */
	public static void trace(String isDetail)
	{
		new NotImplementedException(isDetail).printStackTrace();
	}

	/**
	 * Prints stack trace of unimplemented execution path.
	 *
	 * <p><b>Details:</b> <code>trace()</code> is the same as
	 * <code>trace(String)</code>, except that no detail message is used.</p>
	 *
	 * @see #trace(String)
	 * @since 1999.06.04
	 */
	public static void trace()
	{
		new NotImplementedException().printStackTrace();
	}

}

